<?php

use WPFormsAWeber\Api\Http\Request;
use WPFormsAWeber\Api\Http\Response;
use WPFormsAWeber\Exceptions\HttpMethodNotImplemented;

/**
 * Class WPForms_Aweber_Legacy_Application.
 *
 * The aweber/aweber library is deprecated.
 * Recently it began throwing fatal errors and failing.
 * This class patches some of those issues to keep legacy users working
 * until they can update to V2 (Oauth2) connections.
 * Bypass CurlObject with WPFormsAWeber\Api\Http\Request,
 * and CurlResponse with WPFormsAWeber\Api\Http\Response.
 *
 * @since 2.0.1
 */
class WPForms_Aweber_Legacy_Application extends OAuthApplication {

	/**
	 * Implemented for a standard OAuth adapter interface.
	 *
	 * @since 2.0.1
	 *
	 * @param string $method  The request method.
	 * @param string $uri     The request URI.
	 * @param array  $data    The request data.
	 * @param array  $options The request options.
	 * @param array  $headers The request headers (unused!).
	 *
	 * @return int|mixed|void
	 * @throws AWeberAPIException|HttpMethodNotImplemented|AWeberResponseError  On error.
	 */
	public function request( $method, $uri, $data = [], $options = [], $headers = [] ) {

		$uri = $this->app->removeBaseUri( $uri );
		$url = $this->app->getBaseUri() . $uri;

		$response = $this->makeRequest( $method, $url, $data, $headers );

		if ( ! empty( $options['return'] ) ) {
			if ( $options['return'] === 'status' ) {
				return $response->get_response_code();
			}
			if ( $options['return'] === 'headers' ) {
				return $response->get_headers();
			}
			if ( $options['return'] === 'integer' ) {
				return (int) $response->get_raw_body();
			}
		}

		$data = json_decode( $response->get_raw_body(), true );

		if ( empty( $options['allow_empty'] ) && ! isset( $data ) ) {
			throw new AWeberResponseError( $uri );
		}

		return $data;
	}

	// phpcs:disable PSR2.Methods.MethodDeclaration.Underscore -- Overriding a third-party library method.
	/**
	 * Adds the parameters in associative array $data to the given URL.
	 *
	 * @since 2.0.1
	 *
	 * @param string $url  The URL.
	 * @param array  $data Parameters to be added as a query string to the URL provided.
	 *
	 * @return string
	 */
	protected function _addParametersToUrl( $url, $data ) {

		if ( ! empty( $data ) ) {
			if ( strpos( $url, '?' ) === false ) {
				$url .= '?' . $this->buildData( $data );
			} else {
				$url .= '&' . $this->buildData( $data );
			}
		}

		return $url;
	}
	// phpcs:enable PSR2.Methods.MethodDeclaration.Underscore

	// phpcs:disable Generic.Metrics.CyclomaticComplexity.TooHigh -- Overriding a third-party library method.
	/**
	 * Public facing function to make a request.
	 *
	 * @since  2.0.1
	 *
	 * @param mixed $method  The request method.
	 * @param mixed $url     The request URL.
	 * @param mixed $data    The request data.
	 * @param mixed $headers The request headers (unused).
	 *
	 * @return Response
	 *
	 * @throws AWeberAPIException On error.
	 * @throws HttpMethodNotImplemented On method not implemented.
	 */
	public function makeRequest( $method, $url, $data = [], $headers = [] ) {

		foreach ( $data as $key => $value ) {
			if ( is_array( $value ) ) {
				$data[ $key ] = wp_json_encode( $value );
			}
		}

		switch ( strtoupper( $method ) ) {
			case 'POST':
				$oauth = $this->prepareRequest( $method, $url, $data );
				$resp  = $this->http_post( $url, $oauth );
				break;

			case 'GET':
				$oauth = $this->prepareRequest( $method, $url, $data );
				$resp  = $this->http_get( $url, $oauth );
				break;

			case 'DELETE':
				$oauth = $this->prepareRequest( $method, $url, $data );
				$resp  = $this->http_delete( $url, $oauth );
				break;

			case 'PATCH':
				$oauth = $this->prepareRequest( $method, $url, $data );
				$resp  = $this->http_patch( $url, $oauth );
				break;

			default:
				throw new HttpMethodNotImplemented( 'HTTP method not implemented.' );
		}

		if ( ! $resp ) {
			$msg = 'Unable to connect to the AWeber API.  (' . $this->error . ')';

			$error = [
				'message'           => $msg,
				'type'              => 'APIUnreachableError',
				'documentation_url' => 'https://labs.aweber.com/docs/troubleshooting',
			];

			throw new AWeberAPIException( $error, $url );
		}

		if ( $resp->get_response_code() >= 400 ) {
			$data = json_decode( $resp->get_raw_body(), true );

			throw new AWeberAPIException( $data['error'], $url );
		}

		return $resp;
	}
	// phpcs:enable Generic.Metrics.CyclomaticComplexity.TooHigh

	/**
	 * Make a get request.
	 *
	 * Substitute for OAuthApplication::get().
	 *
	 * @since 2.0.1
	 *
	 * @param string $url  URL to make a get request from.
	 * @param array  $data Data that is used to make the request.
	 *
	 * @access protected
	 * @return Response
	 */
	protected function http_get( $url, $data ) {

		$url = $this->_addParametersToUrl( $url, $data );

		return ( new Request() )->get(
			$url,
			[
				'headers' => $this->get_default_headers(),
				'body'    => null,
			]
		);
	}

	/**
	 * Prepare an OAuth patch method.
	 *
	 * Substitute for OAuthApplication::patch().
	 *
	 * @since 2.0.1
	 *
	 * @param string $url  URL where we are making the request to.
	 * @param array  $data Data that is used to make the request.
	 *
	 * @access protected
	 * @return Response
	 */
	protected function http_patch( $url, $data ) {

		return ( new Request() )->patch(
			$url,
			[
				'headers' => $this->get_default_headers(),
				'body'    => $data,
			]
		);
	}

	/**
	 * Prepare an OAuth post method.
	 *
	 * Substitute for OAuthApplication::post().
	 *
	 * @since 2.0.1
	 *
	 * @param mixed $url  URL where we are making the request to.
	 * @param mixed $data Data that is used to make the request.
	 *
	 * @access protected
	 * @return Response
	 */
	protected function http_post( $url, $data ) {

		return ( new Request() )->post(
			$url,
			[
				'headers' => $this->get_default_headers(),
				'body'    => $data,
			]
		);
	}

	/**
	 * Makes a DELETE request.
	 *
	 * Substitute for OAuthApplication::delete().
	 *
	 * @since 2.0.1
	 *
	 * @param string $url  URL where we are making the request to.
	 * @param array  $data Data that is used to make the request.
	 *
	 * @access protected
	 * @return Response
	 */
	protected function http_delete( $url, $data ) {

		$url = $this->_addParametersToUrl( $url, $data );

		return ( new Request() )->delete(
			$url,
			[
				'headers' => $this->get_default_headers(),
				'body'    => null,
			]
		);
	}

	/**
	 * Parses the body of the response into an array.
	 *
	 * @since 2.0.1
	 *
	 * @param Response $resp The body of a response.
	 *
	 * @return array|mixed|void
	 * @throws AWeberOAuthException On error.
	 */
	public function parseResponse( $resp ) {

		$data = [];

		if ( ! $resp ) {
			return $data;
		}
		if ( empty( $resp ) ) {
			return $data;
		}
		if ( empty( $resp->get_raw_body() ) ) {
			return $data;
		}

		if ( $this->format === 'json' ) {
			$data = json_decode( $resp->get_raw_body() );
		} else {
			parse_str( $resp->get_raw_body(), $data );
		}

		$this->parseAsError( $data );

		return $data;
	}

	/**
	 * Get default headers.
	 *
	 * @since 2.0.1
	 *
	 * @return string[]
	 */
	protected function get_default_headers() {

		return [
			'Accept'       => 'application/json',
			'User-Agent'   => 'wpforms-aweber-legacy/' . WPFORMS_AWEBER_VERSION,
			'Content-Type' => 'application/x-www-form-urlencoded; charset=utf-8',
		];
	}
}
