<?php
// phpcs:ignore Generic.Commenting.DocComment.MissingShort
/** @noinspection AutoloadingIssuesInspection */

/**
 * AWeber integration (Legacy).
 *
 * @since 1.0.0
 */
class WPForms_Aweber extends WPForms_Provider {

	/**
	 * AWeber API Error title string.
	 *
	 * @since 2.0.1
	 */
	const AWEBER_API_ERROR_TITLE = 'AWeber (Legacy) API error';

	/**
	 * AWeber API Error message template.
	 *
	 * @since 2.0.1
	 */
	const AWEBER_API_ERROR_MESSAGE_TEMPLATE = 'API (Legacy) authorization error: %s';

	/**
	 * AWeber Legacy Dependencies.
	 *
	 * Keyed by classname, valued by path.
	 *
	 * @since 2.0.1
	 *
	 * @var array
	 */
	protected $deps = [
		'AWeberAPI'                         => __DIR__ . '/vendor/aweber_api/aweber_api.php',
		'WPForms_Aweber_Legacy_Application' => __DIR__ . '/class-aweber-legacy-application.php',
	];

	/**
	 * Whether the legacy dependencies have been loaded or not.
	 *
	 * @since 2.0.1
	 *
	 * @var bool
	 */
	protected $has_dependencies = false;

	/**
	 * Provider access token.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $access_token;

	/**
	 * Provider access token secret.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $access_token_secret;

	/**
	 * Initialize.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		$this->has_dependencies = $this->load_dependencies();

		$this->version  = WPFORMS_AWEBER_VERSION;
		$this->name     = 'AWeber (Legacy)';
		$this->slug     = 'aweber';
		$this->priority = 10;
		$this->icon     = WPFORMS_AWEBER_URL . 'assets/images/addon-icon-aweber.png';
	}

	/**
	 * Process and submit entry to provider.
	 *
	 * @since 1.0.0
	 *
	 * @param array $fields    Fields.
	 * @param array $entry     Entry.
	 * @param array $form_data Form data.
	 * @param int   $entry_id  Entry id.
	 */
	public function process_entry( $fields, $entry, $form_data, $entry_id = 0 ) { // phpcs:ignore Generic.Metrics.CyclomaticComplexity.MaxExceeded

		// Only run if this form has a connections for this provider.
		if ( empty( $form_data['providers'][ $this->slug ] ) ) {
			return;
		}

		// Fire for each connection. --------------------------------------.

		foreach ( $form_data['providers'][ $this->slug ] as $connection ) :

			// Before proceeding make sure required fields are configured.
			if ( empty( $connection['fields']['Email'] ) ) {
				continue;
			}

			// Setup basic data.
			$full_name  = '';
			$account_id = $connection['account_id'];
			$list_id    = $connection['list_id'];
			$email_data = explode( '.', $connection['fields']['Email'] );
			$email_id   = $email_data[0];
			$email      = $fields[ $email_id ]['value'];
			$data       = [];
			$api        = $this->api_connect( $account_id );

			// Bail if there is any sort of issues with the API connection.
			if ( is_wp_error( $api ) ) {
				continue;
			}

			// Email is required.
			if ( empty( $email ) ) {
				continue;
			}

			// Check for conditionals.
			$pass = $this->process_conditionals( $fields, $entry, $form_data, $connection );

			if ( ! $pass ) {
				wpforms_log(
					'AWeber (Legacy) Subscription stopped by conditional logic',
					$fields,
					[
						'type'    => [ 'provider', 'conditional_logic' ],
						'parent'  => $entry_id,
						'form_id' => $form_data['id'],
					]
				);
				continue;
			}

			// Setup the merge vars. --------------------------------------.

			foreach ( $connection['fields'] as $name => $merge_var ) {

				// Don't include Email or Full name fields.
				if ( $name === 'Email' ) {
					continue;
				}

				// Check if merge var is mapped.
				if ( empty( $merge_var ) ) {
					continue;
				}

				$merge_var = explode( '.', $merge_var );
				$id        = $merge_var[0];
				$key       = ! empty( $merge_var[1] ) ? $merge_var[1] : 'value';

				// Check if mapped form field has a value.
				if ( empty( $fields[ $id ][ $key ] ) ) {
					continue;
				}

				$value = $fields[ $id ][ $key ];

				// Omit the Full Name from custom fields.
				if ( $name === 'Full Name' ) {
					$full_name = $value;

					continue;
				}

				$data[ $name ] = $value;
			}

			$params = [
				'email' => $email,
				'name'  => $full_name,
			];

			if ( ! empty( $data ) ) {
				$params['custom_fields'] = $data;
			}

			// Tags.
			if ( ! empty( $connection['options']['tags'] ) ) {
				$params['tags'] = array_map( 'trim', explode( ',', $connection['options']['tags'] ) );
			}

			// Submit to API. ---------------------------------------------.
			// https://labs.aweber.com/snippets/subscribers.
			try {
				$account     = $this->api[ $account_id ]->getAccount( $this->access_token, $this->access_token_secret );
				$list        = $account->loadFromUrl( "/accounts/{$account->id}/lists/{$list_id}" );
				$subscribers = $list->subscribers;

				$subscribers->create( $params );
			} catch ( AWeberException $e ) {
				wpforms_log(
					'AWeber (Legacy) Subscription error',
					$e->getMessage(),
					[
						'type'    => [ 'provider', 'error' ],
						'parent'  => $entry_id,
						'form_id' => $form_data['id'],
					]
				);
			}

		endforeach;
	}

	/************************************************************************
	 * API methods - these methods interact directly with the provider API. *
	 ************************************************************************/

	/**
	 * Authenticate with the API.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $data    Data.
	 * @param string $form_id Form id.
	 *
	 * @return string|WP_Error Id or error object.
	 */
	public function api_auth( $data = [], $form_id = '' ) {

		list( $consumer_key, $consumer_secret, $access_key, $access_secret, $oauth ) = explode( '|', $data['authcode'] );

		// Connect.
		try {
			$api = $this->get_api_client( $consumer_key, $consumer_secret );
		} catch ( Exception $e ) {
			wpforms_log(
				self::AWEBER_API_ERROR_TITLE,
				$e->getMessage(),
				[
					'type'    => [ 'provider', 'error' ],
					'form_id' => $form_id,
				]
			);

			return $this->error( sprintf( self::AWEBER_API_ERROR_MESSAGE_TEMPLATE, $e->getMessage() ) );
		}

		$api->user->requestToken = $access_key;
		$api->user->tokenSecret  = $access_secret;
		$api->user->verifier     = $oauth;

		// Retrieve an access token.
		try {
			list( $access_token, $access_token_secret ) = $api->getAccessToken();
		} catch ( AWeberException $e ) {
			wpforms_log(
				self::AWEBER_API_ERROR_TITLE,
				$e->getMessage(),
				[
					'type'    => [ 'provider', 'error' ],
					'form_id' => $form_id,
				]
			);

			return $this->error( sprintf( self::AWEBER_API_ERROR_MESSAGE_TEMPLATE, $e->getMessage() ) );
		}

		// Verify we can connect to AWeber.
		try {
			$api->getAccount();
		} catch ( AWeberException $e ) {
			wpforms_log(
				self::AWEBER_API_ERROR_TITLE,
				$e->getMessage(),
				[
					'type'    => [ 'provider', 'error' ],
					'form_id' => $form_id,
				]
			);

			return $this->error( sprintf( self::AWEBER_API_ERROR_MESSAGE_TEMPLATE, $e->getMessage() ) );
		}

		$id    = uniqid( '', true );
		$label = isset( $data['label'] )
			? sanitize_text_field( wp_unslash( $data['label'] ) )
			: $id;

		$providers                       = get_option( 'wpforms_providers', [] );
		$providers[ $this->slug ][ $id ] = [
			'consumer_key'        => $consumer_key,
			'consumer_secret'     => $consumer_secret,
			'access_token'        => $access_token,
			'access_token_secret' => $access_token_secret,
			'label'               => $label,
			'date'                => time(),
		];

		update_option( 'wpforms_providers', $providers );

		return $id;
	}

	/**
	 * Establish connection object to API.
	 *
	 * @since 1.0.0
	 *
	 * @param string $account_id Account id.
	 *
	 * @return mixed AWeberAPI or WP_Error object.
	 */
	public function api_connect( $account_id ) {

		if ( ! empty( $this->api[ $account_id ] ) ) {
			return $this->api[ $account_id ];
		}

		$providers = get_option( 'wpforms_providers' );

		if ( empty( $providers[ $this->slug ][ $account_id ] ) ) {
			return $this->error( 'API error' );
		}

		try {
			$api = $this->get_api_client(
				$providers[ $this->slug ][ $account_id ]['consumer_key'],
				$providers[ $this->slug ][ $account_id ]['consumer_secret']
			);
		} catch ( Exception $e ) {
			wpforms_log(
				self::AWEBER_API_ERROR_TITLE,
				$e->getMessage(),
				[
					'type'       => [ 'provider', 'error' ],
					'account_id' => $account_id,
				]
			);

			return $this->error( sprintf( self::AWEBER_API_ERROR_MESSAGE_TEMPLATE, $e->getMessage() ) );
		}

		$this->api[ $account_id ]  = $api;
		$this->access_token        = $providers[ $this->slug ][ $account_id ]['access_token'];
		$this->access_token_secret = $providers[ $this->slug ][ $account_id ]['access_token_secret'];

		return $api;
	}

	/**
	 * Retrieve provider account lists.
	 *
	 * @since 1.0.0
	 *
	 * @param string $connection_id Connection id.
	 * @param string $account_id    Account id.
	 *
	 * @return mixed Array or WP_Error object.
	 */
	public function api_lists( $connection_id = '', $account_id = '' ) {

		$this->api_connect( $account_id );

		try {
			$account = $this->api[ $account_id ]->getAccount( $this->access_token, $this->access_token_secret );

			if ( ! empty( $account->lists->data['entries'] ) ) {
				return $account->lists->data['entries'];
			}

			return $this->error( 'API list error: No lists found' );
		} catch ( AWeberException $e ) {
			wpforms_log(
				self::AWEBER_API_ERROR_TITLE,
				$e->getMessage(),
				[
					'type' => [ 'provider', 'error' ],
				]
			);

			return $this->error( 'API list error: ' . $e->getMessage() );
		}
	}

	/**
	 * Retrieve provider account list fields.
	 *
	 * @since 1.0.0
	 *
	 * @param string $connection_id Connection id.
	 * @param string $account_id    Account id.
	 * @param string $list_id       List id.
	 *
	 * @return array|WP_Error Array or WP_Error object.
	 */
	public function api_fields( $connection_id = '', $account_id = '', $list_id = '' ) {

		$this->api_connect( $account_id );

		$provider_fields = [
			[
				'name'       => 'Email',
				'field_type' => 'email',
				'req'        => '1',
				'tag'        => 'Email',
			],
			[
				'name'       => 'Full Name',
				'field_type' => 'text',
				'tag'        => 'Full Name',
			],
		];

		try {
			$account = $this->api[ $account_id ]->getAccount( $this->access_token, $this->access_token_secret );
			$fields  = $account->loadFromUrl( "/accounts/{$account->id}/lists/{$list_id}/custom_fields" );
		} catch ( AWeberException $e ) {
			wpforms_log(
				self::AWEBER_API_ERROR_TITLE,
				$e->getMessage(),
				[
					'type' => [ 'provider', 'error' ],
				]
			);

			return $this->error( 'API field error: ' . $e->getMessage() );
		}

		if ( ! empty( $fields->data['entries'] ) ) {
			foreach ( $fields->data['entries'] as $field ) {
				$provider_fields[ $field['name'] ] = [
					'name'       => $field['name'],
					'field_type' => 'text',
					'tag'        => $field['name'],
				];
			}
		}

		return $provider_fields;
	}

	/*************************************************************************
	 * Output methods - these methods generally return HTML for the builder. *
	 *************************************************************************/

	/**
	 * Provider account authorize fields HTML.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function output_auth() {

		$providers = get_option( 'wpforms_providers' );
		$class     = ! empty( $providers[ $this->slug ] ) ? 'hidden' : '';

		$output = '<div class="wpforms-provider-account-add ' . $class . ' wpforms-connection-block">';

		$output .= '<p>';

		$output .= sprintf(
			wp_kses( /* translators: %1$s - Documentation URL. */
				__( 'Adding new AWeber (Legacy) connections has been disabled. Please use the new AWeber integration to add a connection. See our <a class="wpforms-aweber_v2-docs" href="%1$s" rel="noopener" target="_blank">AWeber addon documentation</a> for more details.', 'wpforms-aweber' ),
				[
					'a' => [
						'class'  => [],
						'href'   => [],
						'rel'    => [],
						'target' => [],
					],
				]
			),
			esc_url(
				wpforms_utm_link(
					'https://wpforms.com/docs/install-use-aweber-addon-wpforms/',
					'Builder Settings',
					'AWeber Documentation'
				)
			)
		);

		$output .= '</p>';

		$output .= '</div>';

		return $output;
	}

	/**
	 * Provider account list groups HTML.
	 *
	 * @since 1.0.0
	 *
	 * @param string $connection_id Connection id.
	 * @param array  $connection    Connection.
	 *
	 * @return string
	 */
	public function output_groups( $connection_id = '', $connection = [] ) {

		// No groups or segments for this provider.
		return '';
	}

	/**
	 * Provider account list options HTML.
	 *
	 * @since 1.0.0
	 *
	 * @param string $connection_id Connection id.
	 * @param array  $connection    Connection.
	 *
	 * @return string
	 */
	public function output_options( $connection_id = '', $connection = [] ) {

		if ( empty( $connection_id ) || empty( $connection['account_id'] ) || empty( $connection['list_id'] ) ) {
			return '';
		}

		$output = '<div class="wpforms-provider-options wpforms-connection-block">';

		$output .= '<h4>' . esc_html__( 'Options', 'wpforms-aweber' ) . '</h4>';

		$output .= sprintf(
			'<p>
				<label for="%s_options_tags" class="block">%s <i class="fa fa-question-circle-o wpforms-help-tooltip" title="%s"></i></label>
				<input id="%s_options_tags" type="text" name="providers[%s][%s][options][tags]" value="%s">
			</p>',
			esc_attr( $connection_id ),
			esc_html__( 'Lead Tags', 'wpforms-aweber' ),
			esc_attr__( 'Comma-separated list of tags to assign to a new lead in AWeber.', 'wpforms-aweber' ),
			esc_attr( $connection_id ),
			esc_attr( $this->slug ),
			esc_attr( $connection_id ),
			! empty( $connection['options']['tags'] ) ? esc_attr( $connection['options']['tags'] ) : ''
		);

		$output .= '</div>';

		return $output;
	}

	/*************************************************************************
	 * Integrations tab methods - these methods relate to the settings page. *
	 *************************************************************************/

	/**
	 * Form fields to add a new provider account.
	 *
	 * @since 1.0.0
	 */
	public function integrations_tab_new_form() {}

	/**
	 * Load legacy dependencies.
	 *
	 * @since 2.0.1
	 *
	 * @return bool
	 */
	protected function load_dependencies() {

		// Load dependencies if they are not already loaded.
		foreach ( $this->deps as $classname => $path ) {
			if ( ! class_exists( $classname ) && is_readable( $path ) ) {
				require_once $path;
			}
		}

		// Check if all dependencies are loaded.
		foreach ( array_keys( $this->deps ) as $classname ) {
			if ( ! class_exists( $classname ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Get AWeber API OAuth1 Client.
	 *
	 * @since 2.0.1
	 *
	 * @param string $consumer_key    The consumer key.
	 * @param string $consumer_secret The consumer secret.
	 *
	 * @return AWeberAPI
	 * @throws Exception On missing dependencies, or invalid OAuthServiceProvider.
	 */
	protected function get_api_client( $consumer_key, $consumer_secret ) {

		if ( ! $this->has_dependencies ) {
			throw new Exception( 'AWeber (Legacy) missing dependencies' );
		}

		$api                     = new AWeberAPI( $consumer_key, $consumer_secret );
		$service_provider        = new AWeberServiceProvider();
		$adapter                 = new WPForms_Aweber_Legacy_Application( $service_provider );
		$adapter->consumerKey    = $consumer_key; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		$adapter->consumerSecret = $consumer_secret; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase

		$api->setAdapter( $adapter );

		return $api;
	}

	/**
	 * Add provider to the Settings Integrations tab.
	 *
	 * @since 2.0.1
	 *
	 * @param array $active   Array of active connections.
	 * @param array $settings Array of all connections settings.
	 */
	public function integrations_tab_options( $active, $settings ) {

		$connected = ! empty( $active[ $this->slug ] );
		$accounts  = ! empty( $settings[ $this->slug ] ) ? $settings[ $this->slug ] : [];
		$class     = $connected && $accounts ? 'connected' : '';
		$arrow     = 'right';

		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Nonce verification was handled by Core.
		// This lets us highlight a specific service by a special link.
		if ( isset( $_GET['wpforms-integration'] ) ) {
			if ( $this->slug === $_GET['wpforms-integration'] ) {
				$class .= ' focus-in';
				$arrow  = 'down';
			} else {
				$class .= ' focus-out';
			}
		}
		// phpcs:enable WordPress.Security.NonceVerification.Recommended

		$icon_alt = sprintf( /* translators: %s - addon name. */
			__( '%s addon', 'wpforms-aweber' ),
			$this->name
		);
		?>

		<div id="wpforms-integration-<?php echo esc_attr( $this->slug ); ?>" class="wpforms-settings-provider wpforms-clear <?php echo esc_attr( $this->slug ); ?> <?php echo esc_attr( $class ); ?>">

			<div class="wpforms-settings-provider-header wpforms-clear" data-provider="<?php echo esc_attr( $this->slug ); ?>">

				<div class="wpforms-settings-provider-logo">
					<i title="<?php esc_attr_e( 'Show Accounts', 'wpforms-aweber' ); ?>" class="fa fa-chevron-<?php echo esc_attr( $arrow ); ?>"></i>
					<img src="<?php echo esc_url( $this->icon ); ?>" alt="<?php echo esc_attr( $icon_alt ); ?>">
				</div>

				<div class="wpforms-settings-provider-info">
					<h3><?php echo esc_html( $this->name ); ?></h3>
					<p>
						<?php
						printf( /* translators: %s - provider name. */
							esc_html__( 'Integrate %s with WPForms', 'wpforms-aweber' ),
							esc_html( $this->name )
						);
						?>
					</p>
					<span class="connected-indicator green"><i class="fa fa-check-circle-o"></i>&nbsp;<?php esc_html_e( 'Connected', 'wpforms-aweber' ); ?></span>
				</div>

			</div>

			<div class="wpforms-settings-provider-accounts" id="provider-<?php echo esc_attr( $this->slug ); ?>">

				<div class="wpforms-settings-provider-accounts-list">
					<ul>
						<?php
						foreach ( $accounts as $key => $account ) {
							echo '<li class="wpforms-clear">';
							echo '<span class="label">' . esc_html( $account['label'] ) . '</span>';
							/* translators: %s - connection date. */
							echo '<span class="date">' . sprintf( esc_html__( 'Connected on: %s', 'wpforms-aweber' ), esc_html( date_i18n( get_option( 'date_format' ), (int) $account['date'] ) ) ) . '</span>';
							echo '<span class="remove"><a href="#" data-provider="' . esc_attr( $this->slug ) . '" data-key="' . esc_attr( $key ) . '">' . esc_html__( 'Disconnect', 'wpforms-aweber' ) . '</a></span>';
							echo '</li>';
						}
						?>
					</ul>
				</div>

				<div class="notice notice-warning inline">
					<p style="margin: 0.5em 0;">
						<?php
						printf(
							wp_kses( /* translators: %1$s - Documentation URL. */
								__( 'Adding new AWeber (Legacy) connections has been disabled. Please use the new AWeber integration to add a connection. See our <a class="wpforms-aweber_v2-docs" href="%1$s" rel="noopener" target="_blank">AWeber addon documentation</a> for more details.', 'wpforms-aweber' ),
								[
									'a' => [
										'class'  => [],
										'href'   => [],
										'rel'    => [],
										'target' => [],
									],
								]
							),
							esc_url(
								wpforms_utm_link(
									'https://wpforms.com/docs/install-use-aweber-addon-wpforms/',
									'Settings - Integration',
									'AWeber Documentation'
								)
							)
						);
						?>
					</p>
				</div>

			</div>

		</div>
		<?php
	}
}
