<?php

namespace WPFormsAWeber\Api;

// phpcs:disable WPForms.PHP.UseStatement.UnusedUseStatement
use ArrayAccess;
use Countable;
use Iterator;
use Exception;
use WPFormsAWeber\Exceptions\AccountUpdateException;
use WPFormsAWeber\Exceptions\ApiError;
use WPFormsAWeber\Exceptions\TokenRefreshException;
// phpcs:enable WPForms.PHP.UseStatement.UnusedUseStatement

/**
 * AWeberCollection class.
 *
 * @since 2.0.0
 */
class AWeberCollection extends AWeberResponse implements ArrayAccess, Countable, Iterator {

	/**
	 * The page size.
	 *
	 * @since 2.0.0
	 *
	 * @var int
	 */
	protected $page_size = 100;

	/**
	 * The page start.
	 *
	 * @since 2.0.0
	 *
	 * @var int
	 */
	protected $page_start = 0;

	/**
	 * List of keys that are not publicly accessible.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	protected $private_data = [
		'entries',
		'start',
		'next_collection_link',
	];

	/**
	 * The iteration key.
	 *
	 * @since 2.0.0
	 *
	 * @var int
	 */
	protected $iteration_key = 0;

	/**
	 * Constructor.
	 *
	 * @since 2.0.0
	 *
	 * @param array  $response The response.
	 * @param string $url      The URL.
	 * @param object $adapter  The adapter.
	 */
	public function __construct( $response, $url, $adapter ) {

		parent::__construct( $response, $url, $adapter );
		$this->update_page_size();
	}

	/**
	 * Update the page size.
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 */
	protected function update_page_size() {

		// Grab the url, or prev and next url and pull ws.size from it.
		$url = $this->url;

		if ( array_key_exists( 'next_collection_link', $this->data ) ) {
			$url = $this->data['next_collection_link'];
		} elseif ( array_key_exists( 'prev_collection_link', $this->data ) ) {
			$url = $this->data['prev_collection_link'];
		}

		// Scan querystring for ws_size.
		$url_parts = wp_parse_url( $url );

		// We have a query string.
		if ( array_key_exists( 'query', $url_parts ) ) {
			parse_str( $url_parts['query'], $params );

			// We have a ws_size.
			if ( array_key_exists( 'ws_size', $params ) ) {
				// Set page_size.
				$this->page_size = $params['ws_size'];

				return;
			}
		}

		// We don't have one, just count the # of entries.
		$this->page_size = count( $this->data['entries'] );
	}

	/**
	 * Gets an entry object of this collection type with the given id.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $id ID of the entry you are requesting.
	 *
	 * @return AWeberEntry
	 * @throws ApiError | TokenRefreshException | AccountUpdateException On API error | On token refresh error.
	 */
	public function get_by_id( $id ) {

		$data = $this->adapter->request( 'GET', "{$this->url}/{$id}" );
		$url  = "{$this->url}/{$id}";

		return new AWeberEntry( $data, $url, $this->adapter );
	}

	/**
	 * Gets an entry's parent entry.
	 *
	 * Returns NULL if no parent entry.
	 *
	 * @since 2.0.0
	 *
	 * @return null|AWeberEntry
	 */
	public function get_parent_entry() {

		$url_parts = explode( '/', $this->url );
		$size      = count( $url_parts );

		// Remove collection id and slash from end of url.
		$url = substr( $this->url, 0, - strlen( $url_parts[ $size - 1 ] ) - 1 );

		try {
			$data = $this->adapter->request( 'GET', $url );

			return new AWeberEntry( $data, $url, $this->adapter );
		} catch ( Exception $e ) {
			return null;
		}
	}

	/**
	 * Invoke the API method to CREATE a new entry resource.
	 *
	 * Note: Not all entry resources are eligible to be created, please
	 *       refer to the AWeber API Reference Documentation at
	 *       https://labs.aweber.com/docs/reference/1.0 for more
	 *       details on which entry resources may be created and what
	 *       attributes are required for creating resources.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $kv_pairs Associative array of key/value pairs.
	 *
	 * @return AWeberEntry(Resource) The new resource created
	 * @throws ApiError | TokenRefreshException | AccountUpdateException On API error | On token refresh error.
	 */
	public function create( $kv_pairs ) {

		// Create Resource.
		$params = array_merge( [ 'ws.op' => 'create' ], $kv_pairs );
		$data   = $this->adapter->request( 'POST', $this->url, $params, [ 'return' => 'headers' ] );

		// Return new Resource.
		$url           = $data['Location'];
		$resource_data = $this->adapter->request( 'GET', $url );

		return new AWeberEntry( $resource_data, $url, $this->adapter );
	}

	/**
	 * Interpret what type of resources are held in this collection by analyzing the URL.
	 *
	 * @since 2.0.0
	 *
	 * @return mixed
	 */
	protected function get_type() {

		$url_parts = explode( '/', $this->url );

		return array_pop( $url_parts );
	}

	/**
	 * Invoke the API 'find' operation on a collection to return a subset
	 * of that collection.  Not all collections support the 'find' operation.
	 * refer to https://labs.aweber.com/docs/reference/1.0 for more information.
	 * refer to https://labs.aweber.com/docs/reference/1.0 for a complete list of valid search filters.
	 * filtering on attributes that require additional permissions to display requires an app authorized with those additional permissions.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $search_data Associative array of key/value pairs used as search filters.
	 *
	 * @return AWeberCollection
	 * @throws ApiError | TokenRefreshException | AccountUpdateException | Exception On API error | On token refresh error | On error.
	 */
	public function find( $search_data ) {

		// Invoke find operation.
		$params = array_merge( $search_data, [ 'ws.op' => 'find' ] );
		$data   = $this->adapter->request( 'GET', $this->url, $params );

		// Get total size.
		$ts_params          = array_merge( $params, [ 'ws.show' => 'total_size' ] );
		$total_size         = $this->adapter->request( 'GET', $this->url, $ts_params, [ 'return' => 'integer' ] );
		$data['total_size'] = $total_size;

		// Return collection.
		return $this->read_response( $data, $this->url );
	}

	/**
	 * Fetch the collection data.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $offset The offset to fetch the collection data from.
	 *
	 * @return null|void
	 * @throws ApiError | TokenRefreshException | AccountUpdateException On API error | On token refresh error.
	 */
	protected function fetch_collection_data( $offset ) { // phpcs:ignore Generic.Metrics.CyclomaticComplexity.TooHigh

		// We don't have a next page, we're done.
		if ( ! array_key_exists( 'next_collection_link', $this->data ) ) {
			return null;
		}

		// Snag query string args from collection.
		$parsed = wp_parse_url( $this->data['next_collection_link'] );

		// Parse the query string to get params.
		$pairs = explode( '&', $parsed['query'] );

		// Ensure params is defined.
		$params = [];

		foreach ( $pairs as $pair ) {
			list( $key, $val ) = explode( '=', $pair );

			$params[ $key ] = $val;
		}

		// Calculate new args.
		$limit              = isset( $params['ws.size'] ) ? $params['ws.size'] : 100;
		$pagination_offset  = (int) ( $offset / $limit ) * $limit;
		$params['ws.start'] = $pagination_offset;

		// Fetch data, exclude query string.
		$url_parts        = explode( '?', $this->url );
		$data             = $this->adapter->request( 'GET', $url_parts[0], $params );
		$this->page_start = $params['ws.start'];
		$this->page_size  = isset( $params['ws.size'] ) ? $params['ws.size'] : 100;

		$collection_data = [ 'entries', 'next_collection_link', 'prev_collection_link', 'ws.start' ];

		foreach ( $collection_data as $item ) {
			if ( ! array_key_exists( $item, $this->data ) ) {
				continue;
			}
			if ( ! array_key_exists( $item, $data ) ) {
				continue;
			}
			$this->data[ $item ] = $data[ $item ];
		}
	}

	/**
	 * Assigns a value to the specified offset.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $offset The offset to assign the value to.
	 * @param mixed $value  The value to set.
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function offsetSet( $offset, $value ) {
	}

	/**
	 * Unsets an offset.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $offset The offset to unset.
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function offsetUnset( $offset ) {
	}

	/**
	 * Returns the current element.
	 *
	 * @since 2.0.0
	 *
	 * @return AWeberEntry|null
	 * @throws ApiError | TokenRefreshException | AccountUpdateException On API error | On token refresh error.
	 */
	#[\ReturnTypeWillChange]
	public function current() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		return $this->offsetGet( $this->iteration_key );
	}

	/**
	 * Returns the value at specified offset.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $offset The offset to retrieve.
	 *
	 * @return AWeberEntry|null
	 * @throws ApiError | TokenRefreshException | AccountUpdateException On API error | On token refresh error.
	 */
	#[\ReturnTypeWillChange]
	public function offsetGet( $offset ) {

		if ( ! $this->offsetExists( $offset ) ) {
			return null;
		}

		$limit             = $this->page_size;
		$pagination_offset = (int) ( $offset / $limit ) * $limit;

		// Load collection page if needed.
		if ( $pagination_offset !== $this->page_start ) {
			$this->fetch_collection_data( $offset );
		}

		$entry = $this->data['entries'][ $offset - $pagination_offset ];

		// We have an entry, cast it to an AWeberEntry and return it.
		$entry_url = $this->adapter->app->remove_base_uri( $entry['self_link'] );

		return new AWeberEntry( $entry, $entry_url, $this->adapter );
	}

	/**
	 * Whether or not an offset exists.
	 *
	 * @since 2.0.0
	 *
	 * This method is executed when using isset() or empty()
	 * on objects implementing ArrayAccess.
	 *
	 * @param mixed $offset An offset to check for.
	 *
	 * @return bool
	 */
	#[\ReturnTypeWillChange]
	public function offsetExists( $offset ) { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		return $offset >= 0 && $offset < $this->total_size;
	}

	/**
	 * Moves the current position to the next element.
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function next() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		$this->iteration_key ++;
	}

	/**
	 * Rewinds back to the first element of the Iterator.
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function rewind() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		$this->iteration_key = 0;
	}

	/**
	 * Returns the key of the current element.
	 *
	 * Returns scalar on success, or null on failure.
	 *
	 * @since 2.0.0
	 *
	 * @return int
	 */
	#[\ReturnTypeWillChange]
	public function key() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		return $this->iteration_key;
	}

	/**
	 * Count elements of an object.
	 *
	 * This method is executed when using the count() function on an object implementing Countable.
	 *
	 * @since 2.0.0
	 *
	 * @return int
	 */
	#[\ReturnTypeWillChange]
	public function count() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		return $this->total_size;
	}

	/**
	 * Checks if current position is valid.
	 *
	 * This method is called after Iterator::rewind() and Iterator::next() to check if the current position is valid.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	#[\ReturnTypeWillChange]
	public function valid() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		return $this->offsetExists( $this->key() );
	}
}
