<?php

namespace WPFormsAWeber\Api;

// phpcs:disable WPForms.PHP.UseStatement.UnusedUseStatement
use Exception;
use WPFormsAWeber\Exceptions\AccountUpdateException;
use WPFormsAWeber\Exceptions\ApiError;
use WPFormsAWeber\Exceptions\ResourceNotImplemented;
use WPFormsAWeber\Exceptions\ApiEntryTypeMethodNotImplemented;
use WPFormsAWeber\Exceptions\TokenRefreshException;
// phpcs:enable WPForms.PHP.UseStatement.UnusedUseStatement

/**
 * AWeberEntry class.
 *
 * @since 2.0.0
 */
class AWeberEntry extends AWeberResponse {

	/**
	 * Holds list of data keys that are not publicly accessible.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	protected $private_data = [
		'resource_type_link',
		'http_etag',
	];

	/**
	 * Stores local modifications that have not been saved.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	protected $local_diff = [];

	/**
	 * Holds AWeberCollection objects already instantiated.
	 *
	 * Keyed by their resource name (plural).
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	protected $collections = [];

	/**
	 * Provides a simple array of all the available data (and collections) available in this entry.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function attrs() {

		$attrs = [];

		foreach ( $this->data as $key => $value ) {
			if (
				! in_array( $key, $this->private_data, true )
				&& ! strpos( $key, 'collection_link' )
			) {
				$attrs[ $key ] = $value;
			}
		}

		if ( ! empty( Api::$collection_map[ $this->type ] ) ) {
			foreach ( Api::$collection_map[ $this->type ] as $child ) {
				$attrs[ $child ] = 'collection';
			}
		}

		return $attrs;
	}

	/**
	 * Delete this object from the AWeber system.
	 *
	 * May not be supported by all entry types.
	 *
	 * @since 2.0.0
	 *
	 * @return boolean  Returns true if it is successfully deleted, false
	 *      if the delete request failed.
	 * @throws ApiError | TokenRefreshException | AccountUpdateException On API error | On token refresh error.
	 */
	public function delete() {

		$this->adapter->request(
			'DELETE',
			$this->url,
			[],
			[ 'return' => 'status' ]
		);

		return true;
	}

	/**
	 * Invoke the API method to MOVE an entry resource to a different List.
	 *
	 * Note: Not all entry resources are eligible to be moved, please
	 *       refer to the AWeber API Reference Documentation at
	 *       https://labs.aweber.com/docs/reference/1.0 for more
	 *       details on which entry resources may be moved and if there
	 *       are any requirements for moving that resource.
	 *
	 * @since 2.0.0
	 *
	 * @param AWeberEntry $list          List to move Resource (this) too.
	 * @param int         $last_followup The sequence number of the last followup message sent to the subscriber.
	 *
	 * @return AWeberEntry Resource created on List ($list) or False if resource was not created.
	 * @throws ApiError | TokenRefreshException | AccountUpdateException On API error | On token refresh error.
	 */
	public function move( $list, $last_followup = null ) {

		// Move Resource.
		$params = [
			'ws.op'     => 'move',
			'list_link' => $list->self_link,
		];

		if ( isset( $last_followup ) ) {
			$params['last_followup_message_number_sent'] = $last_followup;
		}

		$data = $this->adapter->request( 'POST', $this->url, $params, [ 'return' => 'headers' ] );

		// Return new Resource.
		$url           = $data['Location'];
		$resource_data = $this->adapter->request( 'GET', $url );

		return new AWeberEntry( $resource_data, $url, $this->adapter );
	}

	/**
	 * Saves the current state of this object if it has been changed.
	 *
	 * @since 2.0.0
	 *
	 * @return true
	 * @throws ApiError | TokenRefreshException | AccountUpdateException On API error | On token refresh error.
	 */
	public function save() {

		if ( ! empty( $this->local_diff ) ) {
			$this->adapter->request(
				'PATCH',
				$this->url,
				$this->local_diff,
				[ 'return' => 'status' ]
			);
		}

		$this->local_diff = [];

		return true;
	}

	/**
	 * Used to look up items in data, and special properties like type and
	 * child collections dynamically.
	 *
	 * @since 2.0.0
	 *
	 * @param string $value Attribute being accessed.
	 *
	 * @return mixed
	 * @throws ResourceNotImplemented | ApiError | TokenRefreshException | AccountUpdateException If the resource is not implemented | On API error | On token refresh error.
	 */
	public function __get( $value ) {

		if ( in_array( $value, $this->private_data, true ) ) {
			return null;
		}

		if ( ! empty( $this->data ) && array_key_exists( $value, $this->data ) ) {
			if ( is_array( $this->data[ $value ] ) ) {
				$array                = new AWeberEntryDataArray( $this->data[ $value ], $value, $this );
				$this->data[ $value ] = $array;
			}

			return $this->data[ $value ];
		}

		if ( $value === 'type' ) {
			return $this->get_type();
		}

		if ( $this->is_child_collection( $value ) ) {
			return $this->get_collection( $value );
		}

		throw new ResourceNotImplemented( "Resource \"{$value}\" is not implemented on this resource." );
	}

	/**
	 * If the key provided is part of the data array, then update it in the
	 * data array.  Otherwise, use the default __set() behavior.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $key   Key of the attr being set.
	 * @param mixed $value Value being set to the $key attr.
	 *
	 * @return void
	 */
	public function __set( $key, $value ) {

		if ( ! array_key_exists( $key, $this->data ) ) {
			parent::__set( $key, $value );

			return;
		}

		$this->local_diff[ $key ] = $value;

		$this->data[ $key ] = $value;
	}

	/**
	 * Used to pull the name of this resource from its resource_type_link.
	 *
	 * @since 2.0.0
	 *
	 * @return String
	 */
	protected function get_type() {

		if ( empty( $this->type ) ) {
			if ( ! empty( $this->data['resource_type_link'] ) ) {
				$this->type = wp_parse_url(
					$this->data['resource_type_link'],
					PHP_URL_FRAGMENT
				);
			} elseif ( ! empty( $this->data['broadcast_id'] ) ) {
				$this->type = 'broadcast';
			} else {
				return null;
			}
		}

		return $this->type;
	}

	/**
	 * Is the given name of a collection a child collection of this entry?
	 *
	 * @since 2.0.0
	 *
	 * @param string $value The name of the collection we are looking for.
	 *
	 * @return bool
	 */
	protected function is_child_collection( $value ) {

		$this->get_type();

		return (
			! empty( Api::$collection_map[ $this->type ] )
			&& in_array( $value, Api::$collection_map[ $this->type ], true )
		);
	}

	/**
	 * Returns the AWeberCollection object representing the given
	 * collection name, relative to this entry.
	 *
	 * @since 2.0.0
	 *
	 * @param string $value The name of the sub-collection.
	 *
	 * @return AWeberCollection
	 * @throws ApiError | TokenRefreshException | AccountUpdateException On API error | On token refresh error.
	 */
	protected function get_collection( $value ) {

		if ( empty( $this->collections[ $value ] ) ) {
			$url                         = "{$this->url}/{$value}";
			$data                        = $this->adapter->request( 'GET', $url );
			$this->collections[ $value ] = new AWeberCollection( $data, $url, $this->adapter );
		}

		return $this->collections[ $value ];
	}

	/**
	 * Looks through all lists for subscribers that match the given filter.
	 *
	 * @since 2.0.0
	 *
	 * @param array $search_data The search data.
	 *
	 * @return AWeberCollection
	 * @throws ApiEntryTypeMethodNotImplemented | ApiError | TokenRefreshException | AccountUpdateException On non-implemented method | On API error | On token refresh error.
	 * @noinspection PhpUnused
	 */
	public function find_subscribers( $search_data ) {

		$this->method_for( [ 'account' ] );
		$params = array_merge( $search_data, [ 'ws.op' => 'findSubscribers' ] );
		$data   = $this->adapter->request( 'GET', $this->url, $params );

		$ts_params  = array_merge( $params, [ 'ws.show' => 'total_size' ] );
		$total_size = $this->adapter->request( 'GET', $this->url, $ts_params, [ 'return' => 'integer' ] );

		// Return collection.
		$data['total_size'] = $total_size;
		$url                = $this->url . '?' . http_build_query( $params );

		return new AWeberCollection( $data, $url, $this->adapter );
	}

	/**
	 * Raises exception if $this->type is not in array entryTypes.
	 * Used to restrict methods to specific entry type(s).
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $entry_types Array of entry types as strings, ie ['account'].
	 *
	 * @return bool
	 * @throws ApiEntryTypeMethodNotImplemented For unsupported entry types.
	 */
	protected function method_for( $entry_types ) {

		if ( in_array( $this->type, $entry_types, true ) ) {
			return true;
		}

		throw new ApiEntryTypeMethodNotImplemented( 'This method is not implemented by the current resource.' );
	}

	/**
	 * Returns analytics activity for a given subscriber.
	 *
	 * @since 2.0.0
	 *
	 * @return AWeberCollection
	 * @throws ApiEntryTypeMethodNotImplemented | ApiError | TokenRefreshException | AccountUpdateException On non-implemented method | On API error | On token refresh error.
	 * @noinspection PhpUnused
	 */
	public function get_activity() {

		$this->method_for( [ 'subscriber' ] );
		$params = [ 'ws.op' => 'getActivity' ];
		$data   = $this->adapter->request( 'GET', $this->url, $params );

		$ts_params  = array_merge( $params, [ 'ws.show' => 'total_size' ] );
		$total_size = $this->adapter->request( 'GET', $this->url, $ts_params, [ 'return' => 'integer' ] );

		// Return collection.
		$data['total_size'] = $total_size;
		$url                = $this->url . '?' . http_build_query( $params );

		return new AWeberCollection( $data, $url, $this->adapter );
	}

	/**
	 * Gets an entry's parent entry.
	 *
	 * Returns NULL if no parent entry
	 *
	 * @since 2.0.0
	 *
	 * @return null|AWeberEntry
	 * @noinspection PhpUnused
	 */
	public function get_parent_entry() {

		$url_parts = explode( '/', $this->url );
		$size      = count( $url_parts );

		// Remove entry id and slash from end of url.
		$url = substr( $this->url, 0, - strlen( $url_parts[ $size - 1 ] ) - 1 );

		// Remove collection name and slash from end of url.
		$url = substr( $url, 0, - strlen( $url_parts[ $size - 2 ] ) - 1 );

		try {
			$data = $this->adapter->request( 'GET', $url );

			return new AWeberEntry( $data, $url, $this->adapter );
		} catch ( Exception $e ) {
			return null;
		}
	}

	/**
	 * Gets all web_forms for this account.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 * @throws ApiEntryTypeMethodNotImplemented | ApiError | TokenRefreshException | AccountUpdateException On non-implemented method | On API error | On token refresh error.
	 * @noinspection PhpUnused
	 */
	public function get_web_forms() {

		$this->method_for( [ 'account' ] );
		$data = $this->adapter->request(
			'GET',
			$this->url . '?ws.op=getWebForms',
			[],
			[ 'allow_empty' => true ]
		);

		return $this->parse_named_operation( $data );
	}

	/**
	 * Turns a dumb array of json into an array of Entries.
	 *
	 * This is NOT a collection, but simply an array of entries,
	 * as returned from a named operation.
	 *
	 * @since 2.0.0
	 *
	 * @param array $data The data.
	 *
	 * @return array
	 */
	protected function parse_named_operation( $data ) {

		$results = [];

		foreach ( $data as $entry_data ) {
			$results[] = new AWeberEntry(
				$entry_data,
				str_replace(
					$this->adapter->app->get_base_uri(),
					'',
					$entry_data['self_link']
				),
				$this->adapter
			);
		}

		return $results;
	}

	/**
	 * Gets all web_form split tests for this account.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 * @throws ApiEntryTypeMethodNotImplemented | ApiError | TokenRefreshException | AccountUpdateException On non-implemented method | On API error | On token refresh error.
	 * @noinspection PhpUnused
	 */
	public function get_web_form_split_tests() {

		$this->method_for( [ 'account' ] );

		$data = $this->adapter->request(
			'GET',
			$this->url . '?ws.op=getWebFormSplitTests',
			[],
			[ 'allow_empty' => true ]
		);

		return $this->parse_named_operation( $data );
	}
}
