<?php

namespace WPFormsAWeber\Api;

use ArrayAccess;
use Countable;
use Iterator;

/**
 * AWeberEntryDataArray class.
 *
 * @since 2.0.0
 */
class AWeberEntryDataArray implements ArrayAccess, Countable, Iterator {

	/**
	 * The data.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	protected $data;

	/**
	 * The keys.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	protected $keys;

	/**
	 * The name.
	 *
	 * @since 2.0.0
	 *
	 * @var mixed
	 */
	protected $name;

	/**
	 * The parent.
	 *
	 * @since 2.0.0
	 *
	 * @var mixed
	 */
	protected $parent;

	/**
	 * The counter.
	 *
	 * @since 2.0.0
	 *
	 * @var int
	 */
	private $counter = 0;

	/**
	 * Constructor.
	 *
	 * @since 2.0.0
	 *
	 * @param array $data   The data.
	 * @param mixed $name   The name.
	 * @param mixed $parent The parent.
	 */
	public function __construct( $data, $name, $parent ) {

		$this->data   = $data;
		$this->keys   = array_keys( $data );
		$this->name   = $name;
		$this->parent = $parent;
	}

	/**
	 * Count elements of an object.
	 *
	 * This method is executed when using the count() function on an object implementing Countable.
	 *
	 * @since 2.0.0
	 *
	 * @return int
	 */
	#[\ReturnTypeWillChange]
	public function count() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		return count( $this->data );
	}

	/**
	 * Whether or not an offset exists.
	 *
	 * @since 2.0.0
	 *
	 * This method is executed when using isset() or empty()
	 * on objects implementing ArrayAccess.
	 *
	 * @param mixed $offset An offset to check for.
	 *
	 * @return bool
	 */
	#[\ReturnTypeWillChange]
	public function offsetExists( $offset ) { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		return ( isset( $this->data[ $offset ] ) );
	}

	/**
	 * Returns the value at specified offset.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $offset The offset to retrieve.
	 *
	 * @return mixed
	 */
	#[\ReturnTypeWillChange]
	public function offsetGet( $offset ) { // phpcs:ignore WPForms.Formatting.EmptyLineAfterFunctionDeclaration.AddEmptyLineAfterFunctionDeclaration

		return $this->data[ $offset ];
	}

	/**
	 * Assigns a value to the specified offset.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $offset The offset to assign the value to.
	 * @param mixed $value  The value to set.
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function offsetSet( $offset, $value ) {

		$this->data[ $offset ]       = $value;
		$this->parent->{$this->name} = $this->data;
	}

	/**
	 * Unsets an offset.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $offset The offset to unset.
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function offsetUnset( $offset ) { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		unset( $this->data[ $offset ] );
	}

	/**
	 * Returns the current element.
	 *
	 * @since 2.0.0
	 *
	 * @return mixed
	 */
	#[\ReturnTypeWillChange]
	public function current() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		return $this->data[ $this->key() ];
	}

	/**
	 * Returns the key of the current element.
	 *
	 * Returns scalar on success, or null on failure.
	 *
	 * @since 2.0.0
	 *
	 * @return mixed
	 */
	#[\ReturnTypeWillChange]
	public function key() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		return $this->keys[ $this->counter ];
	}

	/**
	 * Moves the current position to the next element.
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function next() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		$this->counter ++;
	}

	/**
	 * Rewinds back to the first element of the Iterator.
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 */
	#[\ReturnTypeWillChange]
	public function rewind() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		$this->counter = 0;
	}

	/**
	 * Checks if current position is valid.
	 *
	 * This method is called after Iterator::rewind() and Iterator::next() to check if the current position is valid.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	#[\ReturnTypeWillChange]
	public function valid() { // phpcs:ignore WPForms.Formatting.EmptyLineBeforeReturn.RemoveEmptyLineBeforeReturnStatement

		return ! ( $this->counter >= count( $this->data ) );
	}
}
