<?php

namespace WPFormsAWeber\Api;

use Exception;

/**
 * Api class.
 *
 * @since 2.0.0
 */
class Api {

	/**
	 * Maintains data about what children collections a given object type contains.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	protected static $collection_map = [
		'account'             => [ 'lists', 'integrations' ],
		'broadcast_campaign'  => [ 'links', 'messages', 'stats' ],
		'followup_campaign'   => [ 'links', 'messages', 'stats' ],
		'link'                => [ 'clicks' ],
		'list'                => [
			'campaigns',
			'custom_fields',
			'subscribers',
			'web_forms',
			'web_form_split_tests',
		],
		'web_form'            => [],
		'web_form_split_test' => [ 'components' ],
	];

	/**
	 * The OAuth2 adapter.
	 *
	 * @since 2.0.0
	 *
	 * @var null|OAuth2Adapter
	 */
	public $adapter;

	/**
	 * The account ID.
	 *
	 * @since 2.0.0
	 *
	 * @var null|string
	 */
	private $account_id;

	/**
	 * Constructor.
	 *
	 * @since 2.0.0
	 *
	 * @param string      $access_token  The OAuth2 PKCE access token.
	 * @param string      $refresh_token The OAuth2 PKCE refresh token.
	 * @param int         $expires_on    The OAuth2 PKCE token expiration timestamp.
	 * @param null|string $account_id    The OAuth2 PKCE authenticated user's account ID.
	 */
	public function __construct(
		$access_token,
		$refresh_token,
		$expires_on,
		$account_id = null
	) {

		$this->account_id = $account_id;

		$this->adapter = new OAuth2Adapter(
			$access_token,
			$refresh_token,
			$expires_on,
			$account_id
		);
	}

	/**
	 * Creates an object, either collection or entry, based on the given
	 * URL.
	 *
	 * @since 2.0.0
	 *
	 * @param string $url URL for this request.
	 *
	 * @return mixed|AWeberCollection|AWeberEntry
	 * @throws Exception On error.
	 */
	public function load_from_url( $url ) {

		$data = $this->adapter->request( 'GET', $url );

		return $this->read_response( $data, $url );
	}

	/**
	 * Clean URL.
	 *
	 * @since 2.0.0
	 *
	 * @param string $url The URL to "clean".
	 *
	 * @return string
	 */
	protected function clean_url( $url ) {

		return str_replace( $this->adapter->app->get_base_uri(), '', $url );
	}

	/**
	 * Interprets a response, and creates the appropriate object from it.
	 *
	 * @since 2.0.0
	 *
	 * @param mixed $response Data returned from a request to the AWeberAPI.
	 * @param mixed $url      URL that this data was requested from.
	 *
	 * @return mixed|AWeberCollection|AWeberEntry
	 * @throws Exception On error.
	 */
	protected function read_response( $response, $url ) {

		if ( ! empty( $response['id'] ) || ! empty( $response['broadcast_id'] ) ) {
			return new AWeberEntry( $response, $url, $this->adapter );
		}

		if ( array_key_exists( 'entries', $response ) ) {
			return new AWeberCollection( $response, $url, $this->adapter );
		}

		return $response;
	}

	/**
	 * Retrieve the OAuth2 PKCE authenticated user's account.
	 *
	 * @since 2.0.0
	 *
	 * @link https://api.aweber.com/#tag/Accounts/paths/~1accounts~1%7BaccountId%7D/get
	 *
	 * @return AWeberEntry The entry object associated with the account.
	 * @throws Exception On error.
	 */
	public function get_account() {

		$response_body = $this->adapter->request( 'GET', '/accounts' );
		$accounts      = $this->read_response( $response_body, '/accounts' );

		return $accounts[0];
	}

	/**
	 * Get list tags.
	 *
	 * The aweber/aweber library does not return tags for a list
	 * via the loadFromUrl() method, always returning 'false'.
	 * To workaround that, call $this->adapter->request().instead.
	 *
	 * @since 2.0.0
	 *
	 * @link https://api.aweber.com/#tag/Lists/paths/~1accounts~1%7BaccountId%7D~1lists~1%7BlistId%7D~1tags/get
	 *
	 * @param string $list_id The list ID.
	 *
	 * @return array
	 * @throws Exception On error.
	 */
	public function get_list_tags( $list_id ) {

		if ( $this->account_id === null ) {
			$api_account      = $this->get_account();
			$this->account_id = $api_account->data['id'];
		}

		$response_body = $this->adapter->request( 'GET', "/accounts/{$this->account_id}/lists/{$list_id}/tags" );

		if ( ! is_array( $response_body ) ) {
			return [];
		}

		return $response_body;
	}

	/**
	 * Add a new subscriber to a list.
	 *
	 * @since 2.0.0
	 *
	 * @link https://api.aweber.com/#tag/Subscribers/paths/~1accounts~1%7BaccountId%7D~1lists~1%7BlistId%7D~1subscribers/post
	 *
	 * @param string $list_id Required. The list ID.
	 * @param string $email   Required. The subscriber's email address. [ 1 .. 50 ] characters.
	 * @param array  $args    Optional. Additional arguments to pass to the AWeber API subscribers endpoint.
	 *
	 *     @type array   $custom_fields        The custom fields specified on the subscriber. Note that the custom fields are required to already exist for the list.
	 *     @type string  $name                 The subscriber's name. [ 1 .. 60 ] characters.
	 *     @type string  $update_existing      Enum: "true" "false". If this parameter is present and set to true, then if a subscriber is already present on the list, the subscriber will be updated. Note: Only the fields defined in the patch endpoint will be updated. Any tags in the request will be appended to the existing Subscriber.
	 *     @type array   $tags                 A list of tags to add to the subscriber. This field is used to apply a list of tags to a Subscriber.
	 *
	 * @return array|null
	 * @throws Exception On error.
	 */
	public function add_subscriber_to_list( $list_id, $email, $args = [] ) {

		if ( $this->account_id === null ) {
			$api_account      = $this->get_account();
			$this->account_id = $api_account->data['id'];
		}

		/**
		 * Example...
		 * $data = [
		 *     'email'         => 'jane.doe@example.org',
		 *     'name'          => 'Jane Doe',
		 *     'custom_fields' => [
		 *         'Phone Number' => '+15555551234',
		 *     ],
		 *    'tags'           => [
		 *         'prospect',
		 *     ],
		 * ];
		 */
		$data = [
			'email' => $email,
		];

		if ( ! empty( $args ) && is_array( $args ) ) {
			$data = array_merge( $data, $args );
		}

		$subscribers = $this->load_from_url(
			"/accounts/{$this->account_id}/lists/{$list_id}/subscribers"
		);

		return $subscribers->create( $data );
	}

	/**
	 * Get custom fields for the given list.
	 *
	 * @since 2.0.0
	 *
	 * @param string $list_id The list ID.
	 *
	 * @return array
	 * @throws Exception On error.
	 */
	public function get_custom_fields_for_list( $list_id ) {

		if ( $this->account_id === null ) {
			$api_account      = $this->get_account();
			$this->account_id = $api_account->data['id'];
		}

		$custom_fields_response = $this->load_from_url(
			"/accounts/{$this->account_id}/lists/{$list_id}/custom_fields"
		);

		if ( empty( $custom_fields_response->data['entries'] ) ) {
			return [];
		}

		$fields = [];

		foreach ( $custom_fields_response->data['entries'] as $field ) {
			$fields[ $field['id'] ] = $field['name'];
		}

		return $fields;
	}
}
