<?php

namespace WPFormsAWeber\Api;

use WPFormsAWeber\Plugin;
use WPFormsAWeber\Api\Http\Request;
use WPFormsAWeber\Exceptions\AuthExchangeException;

/**
 * Class ExchangeAuthForToken.
 *
 * Responsible for exchanging the authorization_code and code_verifier
 * for the access_token and refresh_token.
 *
 * @since 2.0.0
 */
class ExchangeAuthForToken {

	/**
	 * Exchange the authorization_code and code_verifier for a token.
	 *
	 * @link https://api.aweber.com/#tag/OAuth-2.0-Reference/paths/~1oauth2~1token/post
	 *
	 * @since 2.0.0
	 *
	 * @param string $authorization_code The authorization code.
	 * @param string $code_verifier      The code verifier.
	 *
	 * @return array
	 * @throws AuthExchangeException On error.
	 */
	public static function exchange( $authorization_code, $code_verifier ) {

		$token_query = TokenQuery::create(
			$authorization_code,
			Plugin::APP_CLIENT_ID,
			$code_verifier
		);

		$request = new Request();

		$response = $request->post( $token_query );

		$response_body = $response->get_body();

		/**
		 * Proof Key for Code Exchange endpoint should always return a 200 on success,
		 * AWeber API should return an error message if the exchange fails on their end.
		 */
		if ( ! isset( $response_body['access_token'] ) || $response->get_response_code() !== 200 ) {
			$message = isset( $response_body['error_description'] )
				? $response_body['error_description']
				: $response->get_response_message();

			throw new AuthExchangeException(
				$message,
				$response->get_response_code()
			);
		}

		/**
		 * Example of a successful token response...
		 * $token = [
		 *     'refresh_token' => 'rand32alphanumcharacterstring123',
		 *     'token_type'    => 'bearer',
		 *     'access_token'  => 'rand32alphanumcharacterstring456',
		 *     'expires_in'    => 7200,
		 * ];.
		 */
		return [
			'refresh_token' => sanitize_text_field( $response_body['refresh_token'] ),
			'token_type'    => sanitize_text_field( $response_body['token_type'] ),
			'access_token'  => sanitize_text_field( $response_body['access_token'] ),
			'expires_in'    => absint( $response_body['expires_in'] ),
			'expires_on'    => time() + absint( $response_body['expires_in'] ),
		];
	}
}
