<?php

namespace WPFormsAWeber\Api\Http;

/**
 * Wrapper class for HTTP requests.
 *
 * @since 2.0.0
 */
class Request {

	/**
	 * Request parameters.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	private $params;

	/**
	 * Request constructor.
	 *
	 * @since 2.0.0
	 *
	 * @param array $params Additional parameters.
	 */
	public function __construct( $params = [] ) {

		$this->params = $params;
	}

	/**
	 * Send a GET request.
	 *
	 * @since 2.0.0
	 *
	 * @param string $url        Request URL.
	 * @param array  $args       Options for the request.
	 * @param array  $query_args Optional query arguments for the URL.
	 *
	 * @return Response
	 */
	public function get( $url, $args = [], $query_args = [] ) {

		if ( ! empty( $query_args ) ) {
			$url = add_query_arg( $query_args, $url );
		}

		return $this->request( 'GET', $url, $args );
	}

	/**
	 * Send a POST request.
	 *
	 * @since 2.0.0
	 *
	 * @param string $url  Request URL.
	 * @param array  $args Options for the request.
	 *
	 * @return Response
	 */
	public function post( $url, $args = [] ) {

		return $this->request( 'POST', $url, $args );
	}

	/**
	 * Send a PATCH request.
	 *
	 * @since 2.0.0
	 *
	 * @param string $url  Request URL.
	 * @param array  $args Options for the request.
	 *
	 * @return Response
	 */
	public function patch( $url, $args = [] ) {

		return $this->request( 'PATCH', $url, $args );
	}

	/**
	 * Send a PUT request.
	 *
	 * @since 2.0.0
	 *
	 * @param string $uri  URI to request.
	 * @param array  $args Options for the request.
	 *
	 * @return Response
	 */
	public function put( $uri, $args = [] ) {

		return $this->request( 'PUT', $uri, $args );
	}

	/**
	 * Send a DELETE request.
	 *
	 * @since 2.0.0
	 *
	 * @param string $url  Request URL.
	 * @param array  $args Options for the request.
	 *
	 * @return Response
	 */
	public function delete( $url, $args = [] ) {

		return $this->request( 'DELETE', $url, $args );
	}

	/**
	 * Send a request based on method (main interface).
	 *
	 * @since 2.0.0
	 *
	 * @see WP_Http::request() For args array structure and WP defaults.
	 *
	 * @param string $method Request method.
	 * @param string $url    Request URL.
	 * @param array  $args   Options for the request.
	 *
	 * @return Response
	 */
	public function request( $method, $url, $args ) {

		// Merge options.
		$options = array_merge_recursive( $this->params, $args );

		// Set a request method.
		$options['method'] = $method;

		/**
		 * Filter a request options array before it's sent.
		 *
		 * @since 2.0.0
		 *
		 * @param array   $options  Request options.
		 * @param string  $method   Request method.
		 * @param string  $url      Request URL.
		 * @param Request $instance Instance of Request class.
		 */
		$options = (array) apply_filters( 'wpforms_aweber_api_http_request_options', $options, $method, $url, $this ); // phpcs:ignore WPForms.PHP.ValidateHooks.InvalidHookName -- aweber should be 1 word, not two.

		// Retrieve the raw response from a safe HTTP request.
		return new Response( wp_safe_remote_request( $url, $options ) );
	}
}
