<?php

namespace WPFormsAWeber\Api\Http;

// phpcs:disable WPForms.PHP.UseStatement.UnusedUseStatement
use WP_Error;
use Traversable;
// phpcs:enable WPForms.PHP.UseStatement.UnusedUseStatement

/**
 * Wrapper class to handle responses.
 *
 * @since 2.0.0
 */
class Response {

	/**
	 * Input data.
	 *
	 * @since 2.0.0
	 *
	 * @var array|WP_Error
	 */
	private $input;

	/**
	 * Response constructor.
	 *
	 * @since 2.0.0
	 *
	 * @param array|WP_Error $input The response data.
	 */
	public function __construct( $input ) {

		$this->input = $input;
	}

	/**
	 * Retrieve only the response code from the raw response.
	 *
	 * @since 2.0.0
	 *
	 * @return int The response code as an integer.
	 */
	public function get_response_code() {

		return absint( wp_remote_retrieve_response_code( $this->input ) );
	}

	/**
	 * Retrieve only the response message from the raw response.
	 *
	 * @since 2.0.0
	 *
	 * @return string The response message.
	 */
	public function get_response_message() {

		$body = $this->get_body();

		if ( is_array( $body ) && ! empty( $body['message'] ) ) {
			return $body['message'];
		}

		$message = wp_remote_retrieve_response_message( $this->input );

		return ! empty( $message ) ? $message : __( 'Response error', 'wpforms-aweber' );
	}

	/**
	 * Retrieve only the body from the raw response.
	 *
	 * @since 2.0.0
	 *
	 * @return array The body of the response.
	 */
	public function get_body() {

		return json_decode( wp_remote_retrieve_body( $this->input ), true );
	}

	/**
	 * Retrieve the body from the raw response without JSON decoding.
	 *
	 * @since 2.0.0
	 *
	 * @return string The body of the response.
	 */
	public function get_raw_body() {

		return wp_remote_retrieve_body( $this->input );
	}

	/**
	 * Get the response headers.
	 *
	 * @since 2.0.0
	 *
	 * @return array|Traversable
	 */
	public function get_headers() {

		return wp_remote_retrieve_headers( $this->input );
	}

	/**
	 * Whether we received errors in the response.
	 *
	 * @since 2.0.0
	 *
	 * @return bool True if response has errors.
	 */
	public function has_errors() {

		$code = $this->get_response_code();

		return $code < 200 || $code > 299;
	}
}
