<?php

namespace WPFormsAWeber\Api;

use WPFormsAWeber\Api\Http\Request;
use WPFormsAWeber\Api\Http\Response;
// phpcs:ignore WPForms.PHP.UseStatement.UnusedUseStatement
use WPFormsAWeber\Exceptions\AccountUpdateException;
use WPFormsAWeber\Exceptions\ApiError;
use WPFormsAWeber\Exceptions\HttpMethodNotImplemented;
use WPFormsAWeber\Exceptions\InvalidRequest;
use WPFormsAWeber\Exceptions\TokenRefreshException;
use WPFormsAWeber\Exceptions\UnauthorizedError;
use WPFormsAWeber\Plugin;
use WPFormsAWeber\Provider\Account;

/**
 * Class OAuth2Adapter.
 *
 * @since 2.0.0
 */
class OAuth2Adapter {

	/**
	 * The user agent string.
	 *
	 * @since 2.0.0
	 *
	 * @var string
	 */
	const USER_AGENT = 'wpforms-aweber/';

	/**
	 * The token base URL.
	 *
	 * @since 2.0.0
	 *
	 * @var string
	 */
	const TOKEN_BASE_URL = 'https://auth.aweber.com/oauth2/token';

	/**
	 * The Content-Type for POST requests.
	 *
	 * Making a POST request to AWeber API
	 * with a Content-Type other than application/x-www-form-urlencode,
	 * will raise a 415 InvalidContentType error.
	 *
	 * @link https://api.aweber.com/#section/API-Errors
	 *
	 * @since 2.0.0
	 *
	 * @var string
	 */
	const CONTENT_TYPE_POST = 'application/x-www-form-urlencoded; charset=utf-8';

	/**
	 * The OAuth2 app.
	 *
	 * @since 2.0.0
	 *
	 * @var null|OAuth2App
	 */
	public $app;

	/**
	 * The OAuth2 PKCE access token.
	 *
	 * @since 2.0.0
	 *
	 * @var null|string
	 */
	private $access_token;

	/**
	 * The OAuth2 PKCE refresh token.
	 *
	 * @since 2.0.0
	 *
	 * @var null|string
	 */
	private $refresh_token;

	/**
	 * The OAuth2 PKCE access token expiration timestamp.
	 *
	 * @since 2.0.0
	 *
	 * @var null|int
	 */
	private $expires_on;

	/**
	 * The account ID.
	 *
	 * @since 2.0.0
	 *
	 * @var null|string
	 */
	private $account_id;

	/**
	 * The account helper.
	 *
	 * If we have a non-empty account ID,
	 * this is used to update the provider options for the account,
	 * when the access token is refreshed.
	 *
	 * @since 2.0.0
	 *
	 * @var null|Account
	 */
	private $account_helper;

	/**
	 * Whether we are currently refreshing the access token.
	 *
	 * @since 2.0.0
	 *
	 * @var bool
	 */
	private $doing_refresh = false;

	/**
	 * Constructor.
	 *
	 * @since 2.0.0
	 *
	 * @param string      $access_token  The OAuth2 PKCE access token.
	 * @param string      $refresh_token The OAuth2 PKCE refresh token.
	 * @param int         $expires_on    The OAuth2 PKCE token expiration timestamp.
	 * @param null|string $account_id    The OAuth2 PKCE authenticated user's account ID.
	 */
	public function __construct(
		$access_token,
		$refresh_token,
		$expires_on,
		$account_id = null
	) {

		$this->access_token   = $access_token;
		$this->refresh_token  = $refresh_token;
		$this->expires_on     = $expires_on;
		$this->account_id     = $account_id;
		$this->app            = new OAuth2App();
		$this->account_helper = new Account();
	}

	/**
	 * Make a request.
	 *
	 * @since 2.0.0
	 *
	 * @param string $method       The method.
	 * @param string $url          The URL.
	 * @param array  $request_body The request body.
	 * @param array  $options      The request options.
	 *
	 * @return array
	 * @throws HttpMethodNotImplemented | ApiError | InvalidRequest | UnauthorizedError On method not implemented | On error | On invalid request | On unauthorized error.
	 */
	private function make_request( $method, $url, $request_body = [], $options = [] ) {

		switch ( strtoupper( $method ) ) {
			case 'POST':
				$response = $this->http_post( $url, $request_body );
				break;

			case 'GET':
				$response = $this->http_get( $url );
				break;

			case 'PATCH':
				$response = $this->http_patch( $url, $request_body );
				break;

			case 'DELETE':
				$response = $this->http_delete( $url, $request_body );
				break;

			default:
				throw new HttpMethodNotImplemented( 'HTTP method not implemented.' );
		}

		// Convert API error messages to Exceptions and bubble up.
		$this->throw_response_errors( $response );

		return $this->handle_response( $response, $options );
	}

	/**
	 * Handle the response.
	 *
	 * @since 2.0.0
	 *
	 * @param Response $response The response object.
	 * @param array    $options  The request options.
	 *
	 * @return mixed
	 */
	private function handle_response( $response, $options = [] ) {

		if ( isset( $options['return'] ) && $options['return'] === 'status' ) {
			return $response->get_response_code();
		}

		if ( isset( $options['return'] ) && $options['return'] === 'headers' ) {
			return $response->get_headers();
		}

		if ( isset( $options['return'] ) && $options['return'] === 'integer' ) {
			return (int) $response->get_raw_body(); // Not decoded.
		}

		return $response->get_body();
	}

	/**
	 * Get default request headers.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	private function get_default_request_headers() {

		$headers = [
			'Accept'     => 'application/json',
			'User-Agent' => self::USER_AGENT . WPFORMS_AWEBER_VERSION,
		];

		if ( $this->doing_refresh === true ) {
			$headers['Content-Type'] = self::CONTENT_TYPE_POST;
		}

		if ( $this->doing_refresh !== true && ! empty( $this->access_token ) ) {
			$headers['Authorization'] = 'Bearer ' . $this->access_token;
		}

		return $headers;
	}

	/**
	 * Make a GET request.
	 *
	 * @since 2.0.0
	 *
	 * @param string $url The URL.
	 *
	 * @return Response
	 */
	private function http_get( $url ) {

		$headers = $this->get_default_request_headers();

		$options = [
			'headers' => $headers,
			'body'    => '',
		];

		return ( new Request() )->get(
			$url,
			$options
		);
	}

	/**
	 * Make a POST request.
	 *
	 * @since 2.0.0
	 *
	 * @param string $url          The URL.
	 * @param array  $request_body The request body.
	 *
	 * @return Response
	 */
	private function http_post( $url, $request_body ) {

		$headers = $this->get_default_request_headers();

		/**
		 * Making a POST request to AWeber API
		 * with a Content-Type other than application/x-www-form-urlencode,
		 * will raise a 415 InvalidContentType error.
		 *
		 * @link https://api.aweber.com/#section/API-Errors
		 */
		$headers['Content-Type'] = self::CONTENT_TYPE_POST;

		$options = [
			'headers' => $headers,
			'body'    => $request_body,
		];

		return ( new Request() )->post(
			$url,
			$options
		);
	}

	/**
	 * Make a PATCH request.
	 *
	 * @since 2.0.0
	 *
	 * @param string $url          The URL.
	 * @param array  $request_body The request body.
	 *
	 * @return Response
	 */
	private function http_patch( $url, $request_body ) {

		$headers = $this->get_default_request_headers();

		$options = [
			'headers' => $headers,
			'body'    => $request_body,
		];

		return ( new Request() )->patch(
			$url,
			$options
		);
	}

	/**
	 * Make a DELETE request.
	 *
	 * @since 2.0.0
	 *
	 * @param string $url          The URL.
	 * @param array  $request_body The request body.
	 *
	 * @return Response
	 */
	private function http_delete( $url, $request_body ) {

		$headers = $this->get_default_request_headers();

		$options = [
			'headers' => $headers,
			'body'    => $request_body,
		];

		return ( new Request() )->delete(
			$url,
			$options
		);
	}

	/**
	 * Request.
	 *
	 * @since 2.0.0
	 *
	 * @param string $method       The request method.
	 * @param string $uri          The request URI.
	 * @param array  $request_body The request body.
	 * @param array  $options      The request options.
	 *
	 * @return array|null
	 * @throws ApiError | TokenRefreshException | AccountUpdateException On API error | On token refresh error.
	 */
	public function request( $method, $uri, $request_body = [], $options = [] ) {

		$this->maybe_refresh_access_token();

		$uri = $this->app->remove_base_uri( $uri );

		$url = $this->app->get_base_uri() . $uri;

		$request_body = $this->serialize_request_body( $request_body );

		for ( $attempt = 0; $attempt < 2; $attempt ++ ) {
			try {
				$response_body = $this->make_request( $method, $url, $request_body, $options );

				break;
			} catch ( UnauthorizedError $exc ) {
				if ( $attempt === 0 ) {
					$this->refresh_access_token();
				} else {
					throw new ApiError( $exc->getMessage(), $exc->getCode() );
				}
			}
		}

		if ( ! isset( $options['allow_empty'] ) && ! isset( $response_body ) ) {
			throw new ApiError( $uri );
		}

		return isset( $response_body ) ? $response_body : null;
	}

	/**
	 * Maybe refresh the access token.
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 * @throws TokenRefreshException | AccountUpdateException On error | On account update error.
	 */
	private function maybe_refresh_access_token() {

		if ( ! $this->is_token_expired() ) {
			return;
		}

		$this->refresh_access_token();
	}

	/**
	 * Check if the access token is expired.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	private function is_token_expired() {

		return ( (int) $this->expires_on < time() );
	}

	/**
	 * Refresh the access token.
	 *
	 * @link https://api.aweber.com/#tag/OAuth-2.0-Reference/paths/~1oauth2~1token/post
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 * @throws TokenRefreshException | AccountUpdateException On error | On account update error.
	 */
	public function refresh_access_token() {

		$this->doing_refresh = true;

		$args = [
			'headers' => [
				'Content-Type' => self::CONTENT_TYPE_POST,
			],
			'body'    => [
				'client_id'     => Plugin::APP_CLIENT_ID,
				'grant_type'    => 'refresh_token',
				'refresh_token' => $this->refresh_token,
			],
		];

		$request = new Request();

		$response = $request->post(
			self::TOKEN_BASE_URL,
			$args
		);

		$this->doing_refresh = false;

		$response_body = $response->get_body();

		/**
		 * Refresh token endpoint should always return a 200 on success,
		 * AWeber API should return an error message if the refresh fails on their end.
		 */
		if ( ! isset( $response_body['access_token'] ) || $response->get_response_code() !== 200 ) {
			$message = isset( $response_body['error_description'] )
				? $response_body['error_description']
				: $response->get_response_message();

			throw new TokenRefreshException(
				$message,
				$response->get_response_code()
			);
		}

		// Set the Token values.
		$this->access_token  = sanitize_text_field( $response_body['access_token'] );
		$this->refresh_token = sanitize_text_field( $response_body['refresh_token'] );
		$this->expires_on    = time() + absint( $response_body['expires_in'] );

		// Update the account token in the provider options.
		if ( ! empty( $this->account_id ) ) {
			$this->account_helper->update_account_token_in_provider_options(
				$this->account_id,
				$this->access_token,
				$this->refresh_token,
				$this->expires_on
			);
		}
	}

	/**
	 * Serialize request body.
	 *
	 * @since 2.0.0
	 *
	 * @param array $request_body The request body.
	 *
	 * @return array
	 */
	private function serialize_request_body( $request_body ) {

		foreach ( $request_body as $key => $value ) {
			if ( is_array( $value ) ) {
				$request_body[ $key ] = wp_json_encode( $value );
			}
		}

		return $request_body;
	}

	/**
	 * Throw response errors.
	 *
	 * @since 2.0.0
	 *
	 * @param Response $response The response object.
	 *
	 * @return void
	 * @throws ApiError On error.
	 * @throws InvalidRequest On invalid request.
	 * @throws UnauthorizedError On unauthorized error.
	 */
	protected function throw_response_errors( $response ) {

		if ( ! $response->has_errors() ) {
			return;
		}

		$response_body = $response->get_body();

		// Expired or invalid token.
		if (
			isset( $response_body['error'] )
			&& $response_body['error'] === 'invalid_token'
			&& $response->get_response_code() === 401
		) {
			throw new UnauthorizedError(
				$response_body['error_description'],
				$response->get_response_code()
			);
		}

		// Invalid request.
		if (
			isset( $response_body['error'] )
			&& $response_body['error'] === 'invalid_request'
			&& $response->get_response_code() === 400
		) {
			throw new InvalidRequest(
				$response_body['error_description'],
				$response->get_response_code()
			);
		}

		if ( isset( $response_body['error_description'] ) ) {
			throw new ApiError(
				$response_body['error_description'],
				$response->get_response_code()
			);
		}

		if ( isset( $response_body['error']['message'] ) ) {
			throw new ApiError(
				$response_body['error']['message'],
				$response->get_response_code()
			);
		}

		throw new ApiError(
			$response->get_response_message(),
			$response->get_response_code()
		);
	}
}
