<?php

namespace WPFormsAWeber;

use stdClass;
use WPForms\Providers\Providers as WPFormsProviders;
use WPForms_Aweber;
use WPForms_Updater;
use WPFormsAWeber\Api\Client;
use WPFormsAWeber\Provider\Account;
use WPFormsAWeber\Provider\Core;
use WPFormsAWeber\Provider\Template;
use WPFormsAWeber\Tasks\ProcessActionTask;

/**
 *  Class Plugin that loads the whole plugin.
 *
 * @since 2.0.0
 */
final class Plugin {

	/**
	 * Provider slug.
	 *
	 * @since 2.0.0
	 *
	 * @var string
	 */
	const SLUG = 'aweber_v2';

	/**
	 * The App Client ID.
	 *
	 * @since 2.0.0
	 *
	 * @var string
	 */
	const APP_CLIENT_ID = 'pehPJJXEF4lBYuR2tjbcnW5gAV2J38aR';

	/**
	 * The Account instance.
	 *
	 * @since 2.0.0
	 *
	 * @var Account
	 */
	private $account;

	/**
	 * The provider templating engine instance.
	 *
	 * @since 2.0.0
	 *
	 * @var Template
	 */
	private $template;

	/**
	 * The API Client instance.
	 *
	 * @since 2.0.0
	 *
	 * @var Client
	 */
	private $client;

	/**
	 * The core provider instance.
	 *
	 * @since 2.0.0
	 *
	 * @var Core
	 */
	private $core;

	/**
	 * Plugin constructor.
	 *
	 * @since 2.0.0
	 */
	public function __construct() {}

	/**
	 * Get the singleton instance of this addon.
	 *
	 * @since 2.0.0
	 *
	 * @return Plugin
	 */
	public static function get_instance() {

		static $instance = null;

		if ( $instance === null ) {
			$instance = new self();

			$instance->init();
		}

		return $instance;
	}

	/**
	 * Get property.
	 *
	 * @since 2.0.0
	 *
	 * @param string $property_name Property name.
	 *
	 * @return mixed
	 */
	public function get( $property_name ) {

		return property_exists( $this, $property_name ) ? $this->{$property_name} : new stdClass();
	}

	/**
	 * All the actual plugin loading is done here.
	 *
	 * @since 2.0.0
	 */
	public function init() {

		$this->hooks();
		$this->init_components();

		// Get all active integrations.
		$providers = wpforms_get_providers_options();

		// Load legacy OAuth1 API integration if the user currently has it set up.
		if ( Helpers::providers_has_oauth1_connection( $providers ) ) {
			require_once WPFORMS_AWEBER_PATH . 'deprecated/class-aweber.php';

			new WPForms_Aweber();
		}

		return $this;
	}

	/**
	 * Hooks.
	 *
	 * @since 2.0.0
	 */
	public function hooks() {

		add_action( 'admin_notices', [ $this, 'upgrade_notice' ] );
		add_action( 'update_option_wpforms_providers', [ $this, 'flush_cache' ] );
		add_action( 'wpforms_updater', [ $this, 'updater' ] );
	}

	/**
	 * Init components.
	 *
	 * @since 2.0.0
	 */
	public function init_components() {

		$this->account  = new Account();
		$this->client   = new Client();
		$this->core     = Core::get_instance();
		$this->template = new Template();

		$this->template->hooks();

		( new ProcessActionTask() )->hooks();

		WPFormsProviders::get_instance()->register(
			$this->core
		);
	}

	/**
	 * Display upgrade notice for sites using the v2 API integration.
	 *
	 * @since 2.0.0
	 */
	public function upgrade_notice() {

		// Only consider showing to admin users.
		if ( ! wpforms_current_user_can() ) {
			return;
		}

		$providers = wpforms_get_providers_options();

		// Only display if site has a legacy OAuth1 API integration configured.
		if ( ! Helpers::providers_has_oauth1_connection( $providers ) ) {
			return;
		}

		?>
		<div class="notice notice-warning wpforms-aweber-update-notice">
			<p>
				<?php esc_html_e( 'Your forms are currently using an outdated AWeber integration that is no longer supported. Please update your forms to use the new integration to avoid losing subscribers.', 'wpforms-aweber' ); ?>
			</p>
			<p>
				<a href="https://wpforms.com/docs/install-use-aweber-addon-wpforms/#update-integration" target="_blank" rel="noopener noreferrer">
					<?php esc_html_e( 'Click here for more details.', 'wpforms-aweber' ); ?>
				</a>
			</p>
		</div>
		<?php
	}

	/**
	 * Flush a transient cache on providers data change.
	 *
	 * @since 2.0.0
	 */
	public function flush_cache() {

		$provider_slug = self::SLUG;

		// Call it when account was added or disconnected.
		if (
			did_action( "wp_ajax_wpforms_settings_provider_add_{$provider_slug}" ) ||
			did_action( "wp_ajax_wpforms_settings_provider_disconnect_{$provider_slug}" )
		) {
			delete_transient( "wpforms_providers_{$provider_slug}_ajax_accounts_get" );
		}
	}

	/**
	 * Load the plugin updater.
	 *
	 * @since 1.0.0
	 *
	 * @param string $key License key.
	 */
	public function updater( $key ) {

		new WPForms_Updater(
			[
				'plugin_name' => 'WPForms AWeber',
				'plugin_slug' => 'wpforms-aweber',
				'plugin_path' => plugin_basename( WPFORMS_AWEBER_FILE ),
				'plugin_url'  => trailingslashit( plugin_dir_url( WPFORMS_AWEBER_URL ) ),
				'remote_url'  => WPFORMS_UPDATER_API,
				'version'     => WPFORMS_AWEBER_VERSION,
				'key'         => $key,
			]
		);
	}
}
