<?php

namespace WPFormsAWeber\Provider;

use Exception;
use WPFormsAWeber\Exceptions\AccountSaveException;
use WPFormsAWeber\Exceptions\AccountUpdateException;
use WPFormsAWeber\Exceptions\RequiredArgumentMissing;
use WPFormsAWeber\Plugin;

/**
 * Class Account.
 *
 * @since 2.0.0
 */
class Account {

	/**
	 * Check if given account ID exists in provider options.
	 *
	 * @since 2.0.0
	 *
	 * @param string $account_id The account ID to check.
	 *
	 * @return bool True if account already exists, else false.
	 */
	public function account_id_exists_in_options( $account_id ) {

		$options = wpforms_get_providers_options(
			Plugin::SLUG
		);

		// If no options exist, the account does not yet exist.
		if ( empty( $options ) ) {
			return false;
		}

		$key = sanitize_key( $account_id );

		// If the key for this account ID does not exist, the account does not yet exist.
		if ( empty( $options[ $key ] ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Get the account details from provider options.
	 *
	 * @since 2.0.0
	 *
	 * @param string $account_id The account ID to check.
	 *
	 * @return array
	 */
	public function get_account_from_options( $account_id ) {

		$options = wpforms_get_providers_options(
			Plugin::SLUG
		);

		// If no options exist, the account does not yet exist.
		if ( empty( $options ) ) {
			return [];
		}

		$key = sanitize_key( $account_id );

		// If the key for this account ID does not exist, the account does not yet exist.
		if ( empty( $options[ $key ] ) ) {
			return [];
		}

		return $options[ $key ];
	}

	/**
	 * Update an existing account's access/refresh tokens in provider options.
	 *
	 * @since 2.0.0
	 *
	 * @param string $account_id    The account ID.
	 * @param string $access_token  The access token.
	 * @param string $refresh_token The refresh token.
	 * @param int    $expires_on    The access token expiration timestamp.
	 *
	 * @return void
	 * @throws AccountUpdateException On attempt to update an account that does not exist in provider options.
	 */
	public function update_account_token_in_provider_options( $account_id, $access_token, $refresh_token, $expires_on ) {

		if ( ! $this->account_id_exists_in_options( $account_id ) ) {
			throw new AccountUpdateException( esc_html__( 'Attempt to update an account that does not exist in provider options.', 'wpforms-aweber' ) );
		}

		$account = $this->get_account_from_options( $account_id );

		$account = array_merge(
			$account,
			[
				'access_token'  => sanitize_text_field( $access_token ),
				'expires_on'    => absint( $expires_on ),
				'refresh_token' => sanitize_text_field( $refresh_token ),
			]
		);

		$key = sanitize_key( $account_id );

		wpforms_update_providers_options(
			Plugin::SLUG,
			$account,
			$key
		);
	}

	/**
	 * Get accounts.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_accounts() {

		$accounts = wpforms_get_providers_options( Plugin::SLUG );

		if ( empty( $accounts ) ) {
			return [];
		}

		return $accounts;
	}

	/**
	 * Remove an account.
	 *
	 * @since 2.0.0
	 *
	 * @param string $account_id Account ID.
	 */
	public function remove_account( $account_id ) {

		$providers = wpforms_get_providers_options();

		if ( empty( $providers[ Plugin::SLUG ][ $account_id ] ) ) {
			return;
		}

		unset( $providers[ Plugin::SLUG ][ $account_id ] );

		update_option( 'wpforms_providers', $providers );
	}

	/**
	 * Get list of available accounts.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_accounts_list() {

		$accounts = $this->get_accounts();

		if ( empty( $accounts ) ) {
			return [];
		}

		foreach ( $accounts as $account_id => $account ) {

			$accounts[ $account_id ] = ! empty( $account['label'] )
				? $account['label']
				: $account_id;
		}

		return $accounts;
	}

	/**
	 * Get API connection.
	 *
	 * @since 2.0.0
	 *
	 * @param string $account_id Account ID.
	 *
	 * @return null|\WPFormsAWeber\Api\Api
	 */
	public function get_connection( $account_id ) {

		$account = $this->get_account_from_options( $account_id );

		if (
			! isset( $account['access_token'], $account['refresh_token'], $account['expires_on'] )
		) {
			return null;
		}

		return wpforms_aweber()
			->get( 'client' )
			->new_connection(
				$account['access_token'],
				$account['refresh_token'],
				$account['expires_on'],
				$account_id
			);
	}

	/**
	 * Check if given account ID has access and refresh tokens.
	 *
	 * @since 2.0.0
	 *
	 * @param string $account_id Account ID.
	 *
	 * @return bool
	 */
	public function account_id_has_token( $account_id ) {

		$options = wpforms_get_providers_options(
			Plugin::SLUG
		);

		// If no options exist, the account does not yet exist.
		if ( empty( $options ) ) {
			return false;
		}

		$key = sanitize_key( $account_id );

		// If the key for this account ID does not exist, the account does not yet exist.
		if ( empty( $options[ $key ] ) ) {
			return false;
		}

		// Missing access or refresh token.
		if (
			empty( $options[ $key ]['access_token'] )
			|| empty( $options[ $key ]['refresh_token'] )
		) {
			return false;
		}

		return true;
	}

	/**
	 * Save new account to provider options.
	 *
	 * @since 2.0.0
	 *
	 * @param string $account_id    The OAuth2 PKCE authenticated user's account ID.
	 * @param string $access_token  The OAuth2 PKCE access token.
	 * @param string $refresh_token The OAuth2 PKCE refresh token.
	 * @param int    $expires_on    The OAuth2 PKCE token expiration timestamp.
	 * @param string $label         The label for the account.
	 *
	 * @return void
	 * @throws AccountSaveException On attempt to save an account that already exists in provider options.
	 */
	public function save_new_account( $account_id, $access_token, $refresh_token, $expires_on, $label = '' ) {

		if ( $this->account_id_exists_in_options( $account_id ) ) {
			throw new AccountSaveException( esc_html__( 'Attempt to save an account that already exists in provider options.', 'wpforms-aweber' ) );
		}

		$key = sanitize_key( $account_id );

		wpforms_update_providers_options(
			Plugin::SLUG,
			[
				'access_token'  => $access_token,
				'account_id'    => $account_id,
				'label'         => ! empty( $label ) ? $label : $account_id,
				'client_id'     => Plugin::APP_CLIENT_ID,
				'date'          => time(),
				'expires_on'    => $expires_on,
				'refresh_token' => $refresh_token,
			],
			$key
		);
	}

	/**
	 * Parse ajax account save data.
	 *
	 * @since 2.0.0
	 *
	 * @param array $data The posted data.
	 *
	 * @return string[]
	 * @throws RequiredArgumentMissing On missing authorization_code or code_verifier.
	 */
	public function parse_ajax_account_save_data( $data ) {

		if ( empty( $data['authorization_code'] ) ) {
			throw new RequiredArgumentMissing( esc_html__( 'Authorization Code', 'wpforms-aweber' ) );
		}

		if ( empty( $data['code_verifier'] ) ) {
			throw new RequiredArgumentMissing( esc_html__( 'Code Verifier', 'wpforms-aweber' ) );
		}

		return [
			'authorization_code' => sanitize_text_field( $data['authorization_code'] ),
			'code_verifier'      => sanitize_text_field( $data['code_verifier'] ),
			'label'              => ! empty( $data['label'] ) ? sanitize_text_field( $data['label'] ) : '',
		];
	}

	/**
	 * Get the remote account details from access_token.
	 *
	 * @since 2.0.0
	 *
	 * @param string $access_token  The access token.
	 * @param string $refresh_token The refresh token.
	 * @param int    $expires_on    The access token expiration timestamp.
	 *
	 * @return array
	 * @throws Exception On error.
	 */
	public function get_remote_account( $access_token, $refresh_token, $expires_on ) {

		$api = wpforms_aweber()
			->get( 'client' )
			->new_connection(
				$access_token,
				$refresh_token,
				$expires_on
			);

		return $api->get_account()->data;
	}

	/**
	 * Sanitize new account details.
	 *
	 * @since 2.0.0
	 *
	 * @param string $account_id    The account ID.
	 * @param string $access_token  The OAuth2 PKCE access token.
	 * @param string $refresh_token The OAuth2 PKCE refresh token.
	 * @param int    $expires_in    The OAuth2 PKCE token expires_in seconds.
	 * @param string $label         The account label. If empty, the account ID will be used.
	 *
	 * @return SingleAccountDTO
	 */
	public function sanitize_new_account_details(
		$account_id,
		$access_token,
		$refresh_token,
		$expires_in,
		$label = ''
	) {

		return new SingleAccountDTO(
			sanitize_text_field( $account_id ),
			sanitize_text_field( $access_token ),
			sanitize_text_field( $refresh_token ),
			time() + absint( $expires_in ),
			! empty( $label ) ? sanitize_text_field( $label ) : sanitize_text_field( $account_id ),
			sanitize_key( sanitize_text_field( $account_id ) )
		);
	}
}
