<?php

namespace WPFormsAWeber\Provider\Actions;

use Exception;
use WPFormsAWeber\Exceptions\RequiredArgumentMissing;

/**
 * Class SubscribeAction.
 *
 * @since 2.0.0
 */
class SubscribeAction extends Action {

	/**
	 * The list ID.
	 *
	 * @since 2.0.0
	 *
	 * @var string
	 */
	protected $list_id;

	/**
	 * The array of args for the AWeber API add subscriber call.
	 *
	 * @since 2.0.0
	 *
	 * @link https://api.aweber.com/#tag/Subscribers/paths/~1accounts~1%7BaccountId%7D~1lists~1%7BlistId%7D~1subscribers/post
	 *
	 * @var array
	 */
	protected $args = [];

	/**
	 * Run action.
	 *
	 * @since 2.0.0
	 *
	 * @see ProcessActionTask::run_action()
	 *
	 * @throws Exception On error.
	 */
	public function run() {

		$this->subscribe_contact();
	}

	/**
	 * Create subscriber.
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 * @throws RequiredArgumentMissing | Exception On missing email and on add subscriber to list.
	 */
	private function subscribe_contact() {

		$this->list_id = $this->connection_data['list_id'];

		$email = $this->field_mapper->get_email();

		if ( empty( $email ) ) {
			throw new RequiredArgumentMissing( 'EMAIL' );
		}

		$this->map();

		$this->connection->add_subscriber_to_list(
			$this->list_id,
			$email,
			$this->args
		);
	}

	/**
	 * Get list of AWeber custom fields for this list.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 * @throws Exception On add subscriber to list.
	 */
	private function get_api_custom_fields() {

		return $this->connection->get_custom_fields_for_list(
			$this->list_id
		);
	}

	/**
	 * Map the non-required arguments for the AWeber API.
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 * @throws Exception On get api custom fields.
	 */
	private function map() {

		$this->map_name();

		$this->map_tags();

		$this->map_custom_fields();
	}

	/**
	 * Map the name field.
	 *
	 * Name is not a required field for adding a subscriber to the AWeber API.
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 */
	private function map_name() {

		$name = $this->field_mapper->get_name();

		if ( wpforms_is_empty_string( $name ) ) {
			return;
		}

		$this->args['name'] = $name;
	}

	/**
	 * Map the tags.
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 */
	private function map_tags() {

		$tags = $this->field_mapper->get_tags();

		if ( count( $tags ) === 0 ) {
			return;
		}

		$this->args['tags'] = $tags;
	}

	/**
	 * Map custom fields.
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 * @throws Exception On get api custom fields.
	 */
	private function map_custom_fields() {

		if ( empty( $this->connection_data['fields_meta'] ) ) {
			return;
		}

		/**
		 * Array of custom fields for the given list.
		 * Keyed by $field['id'], the AWeber ID of the field,
		 * value of $field['name'], the AWeber name for the field.
		 * Example...
		 * $api_fields = [
		 *     1 => 'Phone Number',
		 *     2 => 'Address',
		 * ]
		 */
		$api_fields = $this->get_api_custom_fields();

		if ( empty( $api_fields ) ) {
			return;
		}

		$custom_fields = $this->walk_custom_fields(
			$this->connection_data['fields_meta'],
			$api_fields
		);

		if ( count( $custom_fields ) === 0 ) {
			return;
		}

		$this->args['custom_fields'] = $custom_fields;
	}

	/**
	 * Walk over the custom fields, and map them to the AWeber API fields.
	 *
	 * @since 2.0.0
	 *
	 * @param array $fields_meta The WPForms fields meta.
	 * @param array $api_fields  The AWeber API fields.
	 *
	 * @return array
	 */
	private function walk_custom_fields( $fields_meta, $api_fields ) {

		/**
		 * Example...
		 * $custom_fields = [
		 *     'Phone Number' => '+15555551234',
		 * ]
		 */
		$custom_fields = [];

		if (
			empty( $fields_meta )
			|| empty( $api_fields )
		) {
			return $custom_fields;
		}

		/**
		 * Example...
		 * $field = [
		 *     'name'     => '1', // The API field ID.
		 *     'field_id' => '4', // The WPForms field ID.
		 * ]
		 *
		 * @var array $field
		 */
		foreach ( $fields_meta as $field ) {

			if ( ! isset( $field['name'] ) ) {
				continue;
			}

			$field_id = $this->field_mapper->get_custom_field_value( $field['name'] );

			if ( wpforms_is_empty_string( $field_id ) ) {
				continue;
			}

			$value = $this->field_mapper->get_formatted_value( $field_id, $api_fields[ $field['name'] ] );

			if ( $value === null || wpforms_is_empty_string( $value ) ) {
				continue;
			}

			$custom_fields[ $api_fields[ $field['name'] ] ] = $value;
		}

		return $custom_fields;
	}
}
