<?php

namespace WPFormsAWeber\Provider;

use InvalidArgumentException;
use WPFormsAWeber\Plugin;

/**
 * Class Connection.
 *
 * @since 2.0.0
 */
class Connection {

	/**
	 * Subscribe connection type.
	 *
	 * @since 2.0.0
	 */
	const TYPE_SUBSCRIBE = 'subscribe';

	/**
	 * Connection data.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	private $data = [];

	/**
	 * Constructor method.
	 *
	 * @since 2.0.0
	 *
	 * @param array $raw_data Connection data.
	 *
	 * @throws InvalidArgumentException Emitted when something went wrong.
	 */
	public function __construct( $raw_data ) {

		if ( ! is_array( $raw_data ) || empty( $raw_data ) ) {
			throw new InvalidArgumentException( esc_html__( 'Unexpected connection data.', 'wpforms-aweber' ) );
		}

		$this->set_data( $raw_data );
	}

	/**
	 * Sanitize and set connection data.
	 *
	 * @since 2.0.0
	 *
	 * @param array $raw_data Connection data.
	 */
	protected function set_data( $raw_data ) {

		$this->data = array_replace_recursive( $this->get_required_data(), $raw_data );

		$this->set_id()
		     ->set_connection_name()
		     ->set_form_id()
		     ->set_account_id()
		     ->set_list_id()
		     ->set_email_field()
		     ->set_name_field()
		     ->set_action()
		     ->set_options();
	}

	/**
	 * Sanitize and set connection ID.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function set_id() {

		$this->data['id'] = sanitize_text_field( $this->data['id'] );

		return $this;
	}

	/**
	 * Sanitize and set connection name.
	 *
	 * Note: This is the 'connection_name', not the 'name' field.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function set_connection_name() {

		$this->data['connection_name'] = sanitize_text_field( $this->data['connection_name'] );

		return $this;
	}

	/**
	 * Sanitize and set connection form ID.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function set_form_id() {

		if ( ! wpforms_is_empty_string( $this->data['form_id'] ) ) {
			$this->data['form_id'] = absint( $this->data['form_id'] );
		}

		return $this;
	}

	/**
	 * Sanitize and set connection account ID.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function set_account_id() {

		$this->data['account_id'] = sanitize_text_field( $this->data['account_id'] );

		return $this;
	}

	/**
	 * Sanitize and set connection list ID.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function set_list_id() {

		$this->data['list_id'] = sanitize_text_field( $this->data['list_id'] );

		return $this;
	}

	/**
	 * Sanitize and set email field.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function set_email_field() {

		if (
			isset( $this->data['fields']['EMAIL'] )
			&& ! wpforms_is_empty_string( $this->data['fields']['EMAIL'] )
		) {
			$this->data['fields']['EMAIL'] = absint( $this->data['fields']['EMAIL'] );
		}

		return $this;
	}

	/**
	 * Sanitize and set name field.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function set_name_field() {

		if (
			isset( $this->data['fields']['NAME'] )
			&& ! wpforms_is_empty_string( $this->data['fields']['NAME'] )
		) {
			$this->data['fields']['NAME'] = absint( $this->data['fields']['NAME'] );
		}

		return $this;
	}

	/**
	 * Sanitize and set connection action.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function set_action() {

		if ( ! in_array( $this->data['action'], $this->get_types(), true ) ) {
			$this->data['action'] = '';
		}

		return $this;
	}

	/**
	 * Sanitize and set connection options by action.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function set_options() {

		if ( $this->data['action'] === self::TYPE_SUBSCRIBE ) {
			$this->set_tags()
			     ->set_fields_meta();
		}

		return $this;
	}

	/**
	 * Sanitize and set connection tags.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function set_tags() {

		// Tags already exist in connection data. We just need to sanitize tag names.
		if ( isset( $this->data['tag_names'] ) || isset( $this->data['tag_removes'] ) ) {
			$this->data['tag_names']   = ! empty( $this->data['tag_names'] ) ? array_map( 'sanitize_text_field', (array) $this->data['tag_names'] ) : [];
			$this->data['tag_removes'] = ! empty( $this->data['tag_removes'] ) ? array_map( 'sanitize_text_field', (array) $this->data['tag_removes'] ) : [];

			// Remove unused data in this case.
			unset( $this->data['tags'] );

			return $this;
		}

		return $this->collect_post_tags();
	}

	/**
	 * Collect all tag names.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function collect_post_tags() {

		$add_tags    = $this->collect_add_tags();
		$remove_tags = $this->collect_remove_tags();

		// If user provided new tags.
		if ( isset( $this->data['tags']['new'] ) ) {
			$this->data['tag_news'] = wpforms_chain( $this->data['tags']['new'] )
				->trim()
				->explode( ',' )
				->map(
					static function( $name ) {

						// A tag name cannot be over 100 characters.
						return wp_html_excerpt( sanitize_text_field( $name ), 100 );
					}
				)
				->array_filter(
					static function( $name ) {

						return ! wpforms_is_empty_string( $name );
					}
				)
				->array_merge( $add_tags )
				->array_unique()
				->array_diff( $add_tags )
				->value();
		}

		$this->data['tag_names']   = $add_tags;
		$this->data['tag_removes'] = $remove_tags;

		unset( $this->data['tags'] );

		return $this;
	}

	/**
	 * Collect connection tags which will be added to a contact.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	protected function collect_add_tags() {

		return $this->collect_post_tags_by_source( 'add' );
	}

	/**
	 * Collect connection tags which will be removed from a contact.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	protected function collect_remove_tags() {

		return $this->collect_post_tags_by_source( 'remove' );
	}

	/**
	 * Collect tag names from global $_POST variable.
	 *
	 * @since 2.0.0
	 *
	 * @param string $source_key Source key.
	 *
	 * @return array
	 */
	private function collect_post_tags_by_source( $source_key ) {

		// No tags in connection data - it's a saving process. We need to grab tags from $_POST variable.
		$connection_id = $this->data['id'];
		$form_post     = ! empty( $_POST['data'] ) ? json_decode( stripslashes( $_POST['data'] ), true ) : []; // phpcs:ignore WordPress.Security

		// Native WPForms saving doesn't support multiple selects.
		// We need to get tags data from $_POST variable.
		return wpforms_chain( $form_post )
			->map(
				static function( $post_pair ) use ( $connection_id, $source_key ) {

					$provider_slug = Plugin::SLUG;

					if (
						empty( $post_pair['name'] ) ||
						$post_pair['name'] !== "providers[{$provider_slug}][{$connection_id}][tags][{$source_key}][]"
					) {
						return '';
					}

					return sanitize_text_field( $post_pair['value'] );
				}
			)
			->array_filter()
			->array_values()
			->value();
	}

	/**
	 * Set connection fields meta.
	 *
	 * @since 2.0.0
	 *
	 * @return self
	 */
	protected function set_fields_meta() {

		if ( empty( $this->data['fields_meta'] ) || ! is_array( $this->data['fields_meta'] ) ) {
			$this->data['fields_meta'] = [];

			return $this;
		}

		$this->sanitize_fields_meta();

		return $this;
	}

	/**
	 * Sanitize fields meta.
	 *
	 * @since 2.0.0
	 */
	protected function sanitize_fields_meta() {

		$fields_meta = [];

		foreach ( $this->data['fields_meta'] as $field ) {
			$name     = isset( $field['name'] ) ? sanitize_text_field( $field['name'] ) : '';
			$field_id = isset( $field['field_id'] ) ? sanitize_text_field( $field['field_id'] ) : '';

			if ( wpforms_is_empty_string( $name ) && wpforms_is_empty_string( $field_id ) ) {
				continue;
			}

			$fields_meta[] = [
				'name'     => $name,
				'field_id' => $field_id,
			];
		}

		$this->data['fields_meta'] = $fields_meta;
	}

	/**
	 * Retrieve connection data.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_data() {

		return $this->data;
	}

	/**
	 * Retrieve defaults for connection data.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	protected function get_required_data() {

		return [
			'id'              => false,
			'connection_name' => false,
			'form_id'         => false,
			'account_id'      => false,
			'list_id'         => false,
			'action'          => false,
			'fields'          => [
				'EMAIL' => false,
			],
		];
	}

	/**
	 * Retrieve the list of all connection types.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_types() {

		return [
			self::TYPE_SUBSCRIBE,
		];
	}

	/**
	 * Determine if connection data is valid.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	public function is_valid() {

		foreach ( $this->get_required_data() as $key => $value ) {

			if (
				! isset( $this->data[ $key ] ) ||
				$this->data[ $key ] === false ||
				wpforms_is_empty_string( $this->data[ $key ] )
			) {
				return false;
			}
		}

		return true;
	}
}
