<?php

namespace WPFormsAWeber\Provider;

use WPFormsAWeber\Exceptions\RequiredArgumentMissing;

/**
 * Field Mapper class.
 *
 * @since 2.0.0
 */
class FieldMapper {

	/**
	 * List of fields.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	private $fields;

	/**
	 * Form data.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	private $form_data;

	/**
	 * Connection data.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	private $connection_data;

	/**
	 * Entry ID.
	 *
	 * @since 2.0.0
	 *
	 * @var int
	 */
	private $entry_id;

	/**
	 * FieldMapper constructor.
	 *
	 * @since 2.0.0
	 *
	 * @param array $fields          Array of submitted form fields and values.
	 * @param array $form_data       Form data and settings.
	 * @param array $connection_data Connection data.
	 * @param int   $entry_id        ID of a saved entry.
	 */
	public function __construct( $fields, $form_data, $connection_data, $entry_id ) {

		$this->fields          = $fields;
		$this->form_data       = $form_data;
		$this->connection_data = $connection_data;
		$this->entry_id        = $entry_id;
	}

	/**
	 * Get the email.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 * @throws RequiredArgumentMissing On missing email.
	 */
	public function get_email() {

		if (
			! isset( $this->connection_data['fields']['EMAIL'], $this->fields[ $this->connection_data['fields']['EMAIL'] ] ) ||
			wpforms_is_empty_string( $this->fields[ $this->connection_data['fields']['EMAIL'] ]['value'] )
		) {
			throw new RequiredArgumentMissing( 'EMAIL' );
		}

		return $this->fields[ $this->connection_data['fields']['EMAIL'] ]['value'];
	}

	/**
	 * Get the name.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function get_name() {

		if (
			! isset( $this->connection_data['fields']['NAME'], $this->fields[ $this->connection_data['fields']['NAME'] ] ) ||
			wpforms_is_empty_string( $this->fields[ $this->connection_data['fields']['NAME'] ]['value'] )
		) {
			return '';
		}

		return $this->fields[ $this->connection_data['fields']['NAME'] ]['value'];
	}

	/**
	 * Get new tags.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_new_tags() {

		if ( empty( $this->connection_data['tag_news'] ) ) {
			return [];
		}

		return array_unique( $this->connection_data['tag_news'] );
	}

	/**
	 * Get existing tags.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_existing_tags() {

		if ( empty( $this->connection_data['tag_names'] ) ) {
			return [];
		}

		return array_unique( $this->connection_data['tag_names'] );
	}

	/**
	 * Get both the new and existing tags.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_tags() {

		$new_tags      = $this->get_new_tags();
		$existing_tags = $this->get_existing_tags();

		if (
			count( $new_tags ) === 0
			&& count( $existing_tags ) === 0
		) {
			return [];
		}

		// Merge and remove duplicates.
		return array_unique( array_merge( $new_tags, $existing_tags ) );
	}

	/**
	 * Get field value by field ID.
	 *
	 * @since 2.0.0
	 *
	 * @param int $field_id Field ID.
	 *
	 * @return string
	 */
	public function get_field_value_by_id( $field_id ) {

		if (
			empty( $this->form_data['fields'][ $field_id ] )
			|| ! isset( $this->fields[ $field_id ]['value'] )
		) {
			return '';
		}

		return (string) $this->fields[ $field_id ]['value'];
	}

	/**
	 * Get custom field value.
	 *
	 * @since 2.0.0
	 *
	 * @param string $name Field name.
	 *
	 * @return string
	 */
	public function get_custom_field_value( $name ) {

		$custom_fields = $this->get_custom_fields();

		return isset( $custom_fields[ $name ] ) ? $custom_fields[ $name ] : '';
	}

	/**
	 * Convert custom fields in key => value format.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_custom_fields() {

		if ( empty( $this->connection_data['fields_meta'] ) ) {
			return [];
		}

		return wp_list_pluck( $this->connection_data['fields_meta'], 'field_id', 'name' );
	}

	/**
	 * Get custom field value formatted.
	 *
	 * @since 2.0.0
	 *
	 * @uses FieldMapper::prepare_date_format
	 * @uses FieldMapper::prepare_number_format
	 *
	 * @param int   $field_id Field ID.
	 * @param array $property Property data.
	 *
	 * @return string|null
	 */
	public function get_formatted_value( $field_id, $property ) {

		$type = ! empty( $property['type'] ) ? strtolower( $property['type'] ) : '';

		$mapping_method = 'prepare_' . $type . '_format';

		if ( method_exists( $this, $mapping_method ) ) {

			$value = wpforms_decode_string( $this->$mapping_method( $this->fields[ $field_id ]['value'] ) );

			return str_replace( [ "\r\n", "\r", "\n" ], ', ', $value );
		}

		return wpforms_decode_string( $this->fields[ $field_id ]['value'] );
	}

	/**
	 * Date format.
	 *
	 * @since 2.0.0
	 *
	 * @param string $value Field value.
	 *
	 * @return string|null
	 */
	private function prepare_date_format( $value ) {

		$time = strtotime( $value );

		return $time ? gmdate( 'Y-m-d', $time ) : null;
	}

	/**
	 * Number format.
	 *
	 * @since 2.0.0
	 *
	 * @param string $value Field value.
	 *
	 * @return float|null
	 */
	private function prepare_number_format( $value ) {

		return is_numeric( $value ) ? (float) $value : null;
	}

	/**
	 * Get current entry ID.
	 *
	 * @since 2.0.0
	 *
	 * @return int
	 */
	public function get_entry_id() {

		return $this->entry_id;
	}
}
