<?php

namespace WPFormsAWeber\Provider\Settings;

use Exception;
use WPForms\Providers\Provider\Settings\FormBuilder as FormBuilderAbstract;
use WPFormsAWeber\Api\Api;
use WPFormsAWeber\Api\ExchangeAuthForToken;
use WPFormsAWeber\Plugin;
use WPFormsAWeber\Provider\Connection;
use WPFormsAWeber\Provider\Core;
use WPFormsAWeber\Provider\SingleAccountDTO; // phpcs:ignore WPForms.PHP.UseStatement.UnusedUseStatement

/**
 * Class FormBuilder handles functionality inside the Form Builder.
 *
 * @since 2.0.0
 */
class FormBuilder extends FormBuilderAbstract {

	/**
	 * AWeber API Error title string.
	 *
	 * @since 2.0.1
	 */
	const AWEBER_API_ERROR_TITLE = 'AWeber API error';

	/**
	 * Get the Core loader class of a provider.
	 *
	 * @since 2.0.0
	 *
	 * @var Core
	 */
	protected $core;

	/**
	 * Connections data.
	 *
	 * @since 2.0.0
	 *
	 * @var array
	 */
	private $connections = [];

	/**
	 * Register all hooks (actions and filters).
	 *
	 * This is required by the FormBuilderAbstract
	 * to kick off all the necessary hooks.
	 *
	 * @since 2.0.0
	 */
	protected function init_hooks() {

		parent::init_hooks();

		$this->hooks();
	}

	/**
	 * Register all hooks (actions and filters).
	 *
	 * @since 2.0.0
	 */
	protected function hooks() {

		static $ajax_events = [
			'ajax_account_save',
			'ajax_account_template_get',
			'ajax_accounts_get',
			'ajax_connections_get',
			'ajax_objects_get',
		];

		array_walk(
			$ajax_events,
			static function ( $ajax_event, $key, $instance ) {

				add_filter(
					"wpforms_providers_settings_builder_{$ajax_event}_{$instance->core->slug}",
					[ $instance, $ajax_event ]
				);
			},
			$this
		);

		// Register callbacks for hooks.
		add_filter( 'wpforms_save_form_args', [ $this, 'save_form' ], 11, 2 );
	}

	/**
	 * Pre-process provider data before saving it in form_data when editing a form.
	 *
	 * @since 2.0.0
	 *
	 * @param array $form Form array which is usable with `wp_update_post()`.
	 * @param array $data Data retrieved from $_POST and processed.
	 *
	 * @return array
	 */
	public function save_form( $form, $data ) {

		// Get a filtered (or modified by another addon) form content.
		$form_data = json_decode( stripslashes( $form['post_content'] ), true );

		// Provider exists.
		if ( ! empty( $form_data['providers'][ Plugin::SLUG ] ) ) {
			$modified_post_content = $this->modify_form_data( $form_data );

			if ( ! empty( $modified_post_content ) ) {
				$form['post_content'] = wpforms_encode( $modified_post_content );

				return $form;
			}
		}

		/*
		 * This part works when modification is locked or current filter was called on NOT Providers panel.
		 * Then we need to restore provider connections from the previous form content.
		 */

		// Get a "previous" form content (current content is still not saved).
		$form_obj  = wpforms()->get( 'form' );
		$prev_form = ! empty( $data['id'] ) && $form_obj ? $form_obj->get( $data['id'], [ 'content_only' => true ] ) : [];

		if ( ! empty( $prev_form['providers'][ Plugin::SLUG ] ) ) {
			$provider = $prev_form['providers'][ Plugin::SLUG ];

			if ( ! isset( $form_data['providers'] ) ) {
				$form_data = array_merge( $form_data, [ 'providers' => [] ] );
			}

			$form_data['providers'] = array_merge( (array) $form_data['providers'], [ Plugin::SLUG => $provider ] );
			$form['post_content']   = wpforms_encode( $form_data );
		}

		return $form;
	}

	/**
	 * Prepare modifications for form content, if it's not locked.
	 *
	 * @since 2.0.0
	 *
	 * @param array $form_data Form content.
	 *
	 * @return array|null
	 */
	protected function modify_form_data( $form_data ) {

		/**
		 * Connection is locked.
		 * Why? User clicked the "Save" button when one of the AJAX requests
		 * for retrieval data from API was in progress or failed.
		 */
		if (
			isset( $form_data['providers'][ Plugin::SLUG ]['__lock__'] ) &&
			absint( $form_data['providers'][ Plugin::SLUG ]['__lock__'] ) === 1
		) {
			return null;
		}

		// Modify content as we need, done by reference.
		foreach ( $form_data['providers'][ Plugin::SLUG ] as $connection_id => &$connection ) {

			if ( $connection_id === '__lock__' ) {
				unset( $form_data['providers'][ Plugin::SLUG ]['__lock__'] );
				continue;
			}

			try {
				$connection = ( new Connection( $connection ) )->get_data();
			} catch ( Exception $e ) {
				continue;
			}
		}
		unset( $connection );

		return $form_data;
	}

	/**
	 * Save the data for a new account and validate it.
	 *
	 * @since 2.0.0
	 *
	 * @return array|null
	 */
	public function ajax_account_save() {

		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		$data = wp_unslash( $_POST );

		try {
			$parsed_data = wpforms_aweber()->get( 'account' )->parse_ajax_account_save_data( $data );
		} catch ( Exception $e ) {
			return [
				'error' => esc_html__( 'Please provide a valid verification code.', 'wpforms-aweber' ),
			];
		}

		try {
			$token = ExchangeAuthForToken::exchange( $parsed_data['authorization_code'], $parsed_data['code_verifier'] );
		} catch ( Exception $e ) {
			return [
				'error' => esc_html__( 'Token exchange failed.', 'wpforms-aweber' ),
			];
		}

		try {
			$account = wpforms_aweber()->get( 'account' )->get_remote_account( $token['access_token'], $token['refresh_token'], $token['expires_on'] );
		} catch ( Exception $e ) {
			return [
				'error' => esc_html__( 'Unable to get account details from AWeber API.', 'wpforms-aweber' ),
			];
		}

		if ( wpforms_aweber()->get( 'account' )->account_id_exists_in_options( $account['id'] ) ) {
			return [
				'error' => esc_html__( 'Account already exists.', 'wpforms-aweber' ),
			];
		}

		return $this->save_new_account(
			$account['id'],
			$token['access_token'],
			$token['refresh_token'],
			$token['expires_in'],
			$parsed_data['label']
		);
	}

	/**
	 * Save the new account.
	 *
	 * @since 2.0.0
	 *
	 * @param string $account_id    The account ID.
	 * @param string $access_token  The OAuth2 PKCE access token.
	 * @param string $refresh_token The OAuth2 PKCE refresh token.
	 * @param int    $expires_in    The OAuth2 PKCE token expires_in seconds.
	 * @param string $label         The account label. If empty, the account ID will be used.
	 *
	 * @return array
	 */
	private function save_new_account(
		$account_id,
		$access_token,
		$refresh_token,
		$expires_in,
		$label = ''
	) {
		/**
		 * Sanitized account data.
		 *
		 * @var SingleAccountDTO $sanitized
		 */
		$sanitized = wpforms_aweber()->get( 'account' )->sanitize_new_account_details(
			$account_id,
			$access_token,
			$refresh_token,
			$expires_in,
			$label
		);

		try {
			wpforms_aweber()->get( 'account' )->save_new_account(
				$sanitized->get_account_id(),
				$sanitized->get_access_token(),
				$sanitized->get_refresh_token(),
				$sanitized->get_expires_on(),
				$sanitized->get_label()
			);
		} catch ( Exception $e ) {
			return [
				'error' => esc_html__( 'Unable to save new account.', 'wpforms-aweber' ),
			];
		}

		$this->add_accounts_to_cache(
			[
				$sanitized->get_account_id() => $sanitized->get_label(),
			]
		);

		return [
			'account_id'    => $sanitized->get_account_id(),
			'access_token'  => $sanitized->get_access_token(),
			'refresh_token' => $sanitized->get_refresh_token(),
			'expires_on'    => $sanitized->get_expires_on(),
			'label'         => $sanitized->get_label(),
		];
	}

	/**
	 * Add an array of account keys and labels to the cache.
	 *
	 * @since 2.0.0
	 *
	 * @param array $accounts An array of accounts, where key is account ID and value is account label.
	 *
	 * @return void
	 */
	protected function add_accounts_to_cache( $accounts ) {

		// Get the cache.
		$cache = get_transient( 'wpforms_providers_' . Plugin::SLUG . '_ajax_accounts_get' );

		if ( empty( $cache ) ) {
			$cache = [ 'accounts' => [] ];
		}

		foreach ( $accounts as $key => $label ) {
			$cache['accounts'][ $key ] = $label;
		}

		set_transient( 'wpforms_providers_' . Plugin::SLUG . '_ajax_accounts_get', $cache, 12 * HOUR_IN_SECONDS );
	}

	/**
	 * Content for the "Add New Account" modal.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function ajax_account_template_get() {

		$content = wpforms_aweber()
			->get( 'template' )
			->get_settings_template(
				'new-account-form',
				[
					'core_name' => esc_html__( 'AWeber', 'wpforms-aweber' ),
				]
			);

		return [
			'title'   => esc_html__( 'New AWeber Account', 'wpforms-aweber' ),
			'content' => $content,
			'type'    => 'blue',
		];
	}

	/**
	 * Get the list of all saved connections.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function ajax_connections_get() {

		$connections = [
			'connections'  => ! empty( $this->get_connections_data() ) ? array_reverse( $this->get_connections_data(), true ) : [],
			'conditionals' => [],
		];

		foreach ( $connections['connections'] as $connection ) {
			if ( empty( $connection['id'] ) ) {
				continue;
			}

			// This will either return an empty placeholder or complete set of rules, as a DOM.
			$connections['conditionals'][ $connection['id'] ] = wpforms_conditional_logic()
				->builder_block(
					[
						'form'       => $this->form_data,
						'type'       => 'panel',
						'parent'     => 'providers',
						'panel'      => Plugin::SLUG,
						'subsection' => $connection['id'],
						'reference'  => esc_html__( 'Marketing provider connection', 'wpforms-aweber' ),
					],
					false
				);
		}

		$accounts = $this->ajax_accounts_get();

		return array_merge( $connections, $accounts );
	}

	/**
	 * Get the list of all accounts.
	 *
	 * @since 2.0.0
	 *
	 * @return array May return an empty sub-array.
	 */
	public function ajax_accounts_get() {

		// Get the cache.
		$cache = get_transient( 'wpforms_providers_' . Plugin::SLUG . '_ajax_accounts_get' );

		// Retrieve accounts from cache.
		if ( is_array( $cache ) && isset( $cache['accounts'] ) ) {
			return $cache;
		}

		// If no cache - prepare to make real external requests.
		$data             = [];
		$data['accounts'] = $this->get_accounts_data();

		// Save accounts to cache.
		if ( ! empty( $data['accounts'] ) ) {
			$this->add_accounts_to_cache( $data['accounts'] );
		}

		return $data;
	}

	/**
	 * Retrieve other data (lists, tags, custom fields), that is needed for the process.
	 *
	 * @since 2.0.0
	 *
	 * @return array|null Return null on any kind of error. Array of data otherwise.
	 * @throws Exception On error.
	 */
	public function ajax_objects_get() {

		if ( ! $this->should_ajax_objects_get_run() ) {
			return null;
		}

		// phpcs:disable WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput
		$account_id = sanitize_text_field( wp_unslash( $_POST['account_id'] ) );
		$sources    = array_map( 'wp_validate_boolean', wp_unslash( $_POST['sources'] ) );
		$list_id    = ! empty( $_POST['list_id'] ) ? sanitize_text_field( wp_unslash( $_POST['list_id'] ) ) : '';
		// phpcs:enable WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput

		$account = wpforms_aweber()->get( 'account' )->get_account_from_options( $account_id );

		if ( empty( $account ) ) {
			wpforms_log(
				self::AWEBER_API_ERROR_TITLE,
				'Account not found in options',
				[
					'type' => [ 'provider', 'error' ],
				]
			);

			return null;
		}

		try {
			$api = wpforms_aweber()->get( 'client' )->new_connection(
				$account['access_token'],
				$account['refresh_token'],
				$account['expires_on'],
				$account['account_id']
			);
		} catch ( Exception $e ) {
			wpforms_log(
				self::AWEBER_API_ERROR_TITLE,
				$e->getMessage(),
				[
					'type' => [ 'provider', 'error' ],
				]
			);

			return null;
		}

		return $this->ajax_objects_get_sources( $sources, $account_id, $api, $list_id );
	}

	/**
	 * Hydrate sources.
	 *
	 * @since 2.0.0
	 *
	 * @param array  $sources    The sources array.
	 * @param string $account_id The account ID.
	 * @param Api    $api        The API instance.
	 * @param string $list_id    The list ID.
	 *
	 * @return array
	 * @throws Exception On error.
	 */
	private function ajax_objects_get_sources( $sources, $account_id, $api, $list_id = '' ) {

		// Retrieve lists.
		if ( isset( $sources['lists'] ) ) {
			$sources['lists'] = $this->get_api_object_lists( $api );
		}

		// Retrieve tags.
		if ( isset( $sources['tags'] ) && ! empty( $list_id ) ) {
			$sources['tags'] = $this->get_api_object_list_tags( $api, $list_id );
		}

		// Retrieve custom fields.
		if ( isset( $sources['mergeFields'] ) && ! empty( $list_id ) ) {
			$sources['mergeFields'] = $this->get_api_object_list_fields( $api, $list_id, $account_id );
		}

		return $sources;
	}

	/**
	 * Should the ajax_objects_get() method be run.
	 *
	 * @since 2.0.0
	 *
	 * @return bool
	 */
	private function should_ajax_objects_get_run() {

		$options = $this->core->get_provider_options();

		// phpcs:disable WordPress.Security.NonceVerification
		return ! (
			empty( $options )
			|| empty( $_POST['account_id'] )
			|| empty( $_POST['connection_id'] )
			|| empty( $_POST['sources'] )
			|| empty( $options[ sanitize_key( wp_unslash( $_POST['account_id'] ) ) ] )
		);
		// phpcs:enable WordPress.Security.NonceVerification
	}

	/**
	 * Get lists.
	 *
	 * @since 2.0.0
	 *
	 * @param Api $api Api instance.
	 *
	 * @return array
	 * @throws Exception On error.
	 */
	protected function get_api_object_lists( $api ) {

		try {
			$api_account = $api->get_account();
		} catch ( Exception $e ) {
			wpforms_log(
				self::AWEBER_API_ERROR_TITLE,
				$e->getMessage(),
				[
					'type' => [ 'provider', 'error' ],
				]
			);

			return [];
		}

		$lists = [];

		foreach ( $api_account->lists as $list ) {
			$lists[] = [
				'id'   => (string) $list->id,
				'name' => $list->name,
			];
		}

		return $lists;
	}

	/**
	 * Get list tags.
	 *
	 * @since 2.0.0
	 *
	 * @param Api    $api     Api instance.
	 * @param string $list_id List ID.
	 *
	 * @return array
	 */
	protected function get_api_object_list_tags( $api, $list_id ) {

		try {
			$tags = $api->get_list_tags( $list_id );
		} catch ( Exception $e ) {
			wpforms_log(
				self::AWEBER_API_ERROR_TITLE,
				$e->getMessage(),
				[
					'type' => [ 'provider', 'error' ],
				]
			);

			return [];
		}

		if ( empty( $tags ) || ! is_array( $tags ) ) {
			return [];
		}

		// Remove duplicates.
		return array_unique( $tags );
	}

	/**
	 * Get list fields.
	 *
	 * @since 2.0.0
	 *
	 * @param Api    $api        Api instance.
	 * @param string $list_id    List ID.
	 * @param string $account_id Account ID.
	 *
	 * @return array
	 * @noinspection PhpUnusedParameterInspection
	 */
	protected function get_api_object_list_fields( $api, $list_id, $account_id ) {

		$provider_fields = [
			'EMAIL' => 'Email',
			'NAME'  => 'Name',
		];

		try {
			$custom_fields = $api->get_custom_fields_for_list( $list_id );
		} catch ( Exception $e ) {
			wpforms_log(
				self::AWEBER_API_ERROR_TITLE,
				$e->getMessage(),
				[
					'type' => [ 'provider', 'error' ],
				]
			);

			return [
				'required' => $provider_fields,
				'optional' => [],
			];
		}

		return [
			'required' => $provider_fields,
			'optional' => $custom_fields,
		];
	}

	/**
	 * Retrieve saved provider connections data.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_connections_data() {

		if ( ! isset( $this->form_data['providers'][ Plugin::SLUG ] ) ) {
			return [];
		}

		if ( ! empty( $this->connections ) ) {
			return $this->connections;
		}

		foreach ( $this->form_data['providers'][ Plugin::SLUG ] as &$connection ) {
			$connection = (array) $connection;

			try {
				$connection = ( new Connection( $connection ) )->get_data();
			} catch ( Exception $e ) {
				continue;
			}
		}
		unset( $connection );

		$this->connections = $this->form_data['providers'][ Plugin::SLUG ];

		return $this->connections;
	}

	/**
	 * Retrieve saved provider accounts data.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_accounts_data() {

		return wpforms_aweber()->get( 'account' )->get_accounts_list();
	}

	/**
	 * Display generated fields with all markup for using in provider's connection.
	 * Used internally in templates.
	 *
	 * @since 2.0.0
	 *
	 * @return array
	 */
	public function get_field_html() {

		return [
			'email' => wpforms_panel_field(
				'select',
				Plugin::SLUG,
				'EMAIL',
				$this->form_data,
				esc_html__( 'Subscriber Email', 'wpforms-aweber' ),
				[
					'after'         => '<p class="description">' . esc_html__( "Required. Please select the Email field containing the subscriber's email address.", 'wpforms-aweber' ) . '</p>',
					'after_tooltip' => '<span class="required">*</span>',
					'field_map'     => [ 'email', 'text' ],
					'field_name'    => 'providers[' . Plugin::SLUG . '][%connection_id%][fields][EMAIL]',
					'input_class'   => 'wpforms-required',
					'input_id'      => 'wpforms-panel-field-' . Plugin::SLUG . '-%connection_id%-email',
					'parent'        => 'providers',
					'placeholder'   => esc_html__( '--- Select Email Field ---', 'wpforms-aweber' ),
				],
				false
			),
			'name'  => wpforms_panel_field(
				'select',
				Plugin::SLUG,
				'NAME',
				$this->form_data,
				esc_html__( 'Subscriber Name', 'wpforms-aweber' ),
				[
					'after'       => '<p class="description">' . esc_html__( "Optional. Please select the Name field containing the subscriber's name.", 'wpforms-aweber' ) . '</p>',
					'field_map'   => [ 'name', 'text' ],
					'field_name'  => 'providers[' . Plugin::SLUG . '][%connection_id%][fields][NAME]',
					'input_id'    => 'wpforms-panel-field-' . Plugin::SLUG . '-%connection_id%-name',
					'parent'      => 'providers',
					'placeholder' => esc_html__( '--- Select Name Field ---', 'wpforms-aweber' ),
				],
				false
			),
		];
	}

	/**
	 * Use this method to register own templates for the form builder.
	 * Make sure to have `tmpl-` in template name in `<script id="tmpl-*">`.
	 *
	 * @since 2.0.0
	 */
	public function builder_custom_templates() {

		?>

		<!-- Single AWeber connection. -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( Plugin::SLUG ); ?>-builder-content-connection">
			<?php echo wpforms_aweber()->get( 'template' )->get_builder_template( 'connection' ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
		</script>

		<!-- Single AWeber connection block: general data -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( Plugin::SLUG ); ?>-builder-content-connection-data">
			<?php echo wpforms_aweber()->get( 'template' )->get_builder_template( 'general' ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
		</script>

		<!-- Single AWeber connection block: SUBSCRIBE -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( Plugin::SLUG ); ?>-builder-content-connection-subscribe">
			<?php echo wpforms_aweber()->get( 'template' )->get_builder_template( 'subscribe', [ 'fields' => $this->get_field_html() ] ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
		</script>

		<!-- Single AWeber connection block: REQUIRED FIELDS -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( Plugin::SLUG ); ?>-builder-content-connection-required-fields">
			<?php echo wpforms_aweber()->get( 'template' )->get_builder_template( 'required-fields' ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
		</script>

		<!-- Single AWeber connection block: ERROR -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( Plugin::SLUG ); ?>-builder-content-connection-error">
			<?php echo wpforms_aweber()->get( 'template' )->get_builder_template( 'error' ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
		</script>

		<!-- Single AWeber connection block: LOCK -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( Plugin::SLUG ); ?>-builder-content-connection-lock">
			<?php echo wpforms_aweber()->get( 'template' )->get_builder_template( 'lock' ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
		</script>

		<?php
	}

	/**
	 * Enqueue JavaScript file(s).
	 *
	 * @since 2.0.0
	 */
	public function enqueue_assets() {

		parent::enqueue_assets();

		$min = wpforms_get_min_suffix();

		wp_enqueue_script(
			'wpforms-aweber-admin-builder',
			WPFORMS_AWEBER_URL . "assets/js/aweber-builder{$min}.js",
			[ 'wpforms-admin-builder-providers', 'choicesjs' ],
			WPFORMS_AWEBER_VERSION,
			true
		);

		wp_localize_script(
			'wpforms-aweber-admin-builder',
			'wpformsAWeberBuilderVars',
			[
				'i18n' => [
					'generalAjaxError'    => esc_html__( 'Something went wrong while performing an AJAX request.', 'wpforms-aweber' ),
					'nameFieldFormats'    => [
						'full'   => esc_html__( 'Full', 'wpforms-aweber' ),
						'first'  => esc_html__( 'First', 'wpforms-aweber' ),
						'middle' => esc_html__( 'Middle', 'wpforms-aweber' ),
						'last'   => esc_html__( 'Last', 'wpforms-aweber' ),
					],
					'providerPlaceholder' => esc_html__( '--- Select AWeber Field ---', 'wpforms-aweber' ),
				],
			]
		);
	}
}
