<?php

namespace WPFormsAWeber\Provider\Settings;

use Exception;
use WPForms\Providers\Provider\Settings\PageIntegrations as PageIntegrationsAbstract;
use WPFormsAWeber\Api\ExchangeAuthForToken;
use WPFormsAWeber\Provider\SingleAccountDTO; // phpcs:ignore WPForms.PHP.UseStatement.UnusedUseStatement

/**
 * Class PageIntegrations handles functionality inside the Settings > Integrations page.
 *
 * @since 2.0.0
 */
class PageIntegrations extends PageIntegrationsAbstract {

	/**
	 * AJAX to add a provider from the Settings > Integrations tab.
	 *
	 * @since 2.0.0
	 */
	public function ajax_connect() {

		parent::ajax_connect();

		// phpcs:ignore WordPress.Security -- Parent class checks nonce and current user permissions.
		$data = wp_unslash( wp_parse_args( $_POST['data'] ) );

		try {
			$parsed_data = wpforms_aweber()->get( 'account' )->parse_ajax_account_save_data( $data );
		} catch ( Exception $e ) {
			wp_send_json_error( [ 'error_msg' => $e->getMessage() ] );
		}

		try {
			$token = ExchangeAuthForToken::exchange( $parsed_data['authorization_code'], $parsed_data['code_verifier'] );
		} catch ( Exception $e ) {
			wp_send_json_error( [ 'error_msg' => $e->getMessage() ] );
		}

		try {
			$account = wpforms_aweber()->get( 'account' )->get_remote_account( $token['access_token'], $token['refresh_token'], $token['expires_on'] );
		} catch ( Exception $e ) {
			wp_send_json_error( [ 'error_msg' => $e->getMessage() ] );
		}

		if ( wpforms_aweber()->get( 'account' )->account_id_exists_in_options( $account['id'] ) ) {
			wp_send_json_error( [ 'error_msg' => esc_html__( 'Account with this ID has been already added.', 'wpforms-aweber' ) ] );
		}

		/**
		 * Sanitized account data.
		 *
		 * @var SingleAccountDTO $sanitized
		 */
		$sanitized = wpforms_aweber()->get( 'account' )->sanitize_new_account_details(
			$account['id'],
			$token['access_token'],
			$token['refresh_token'],
			$token['expires_in'],
			$parsed_data['label']
		);

		try {
			wpforms_aweber()->get( 'account' )->save_new_account(
				$sanitized->get_account_id(),
				$sanitized->get_access_token(),
				$sanitized->get_refresh_token(),
				$sanitized->get_expires_on(),
				$sanitized->get_label()
			);
		} catch ( Exception $e ) {
			wp_send_json_error( [ 'error_msg' => $e->getMessage() ] );
		}

		// Success.
		wp_send_json_success( [ 'html' => $this->prepare_result_html_list( $sanitized->get_account_id(), $sanitized->get_label() ) ] );
	}

	/**
	 * Prepare HTML for a new account.
	 *
	 * @since 2.0.0
	 *
	 * @param string $account_id The account ID.
	 * @param string $label      The account label.
	 *
	 * @return string
	 */
	protected function prepare_result_html_list( $account_id, $label ) {

		return wpforms_aweber()->get( 'template' )->get_settings_template(
			'account-connected',
			[
				'label'     => $label,
				'core_slug' => $this->core->slug,
				'key'       => sanitize_key( $account_id ),
			]
		);
	}

	/**
	 * Display fields that will store account details.
	 *
	 * @since 2.0.0
	 */
	protected function display_add_new_connection_fields() {

		// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped
		echo wpforms_aweber()->get( 'template' )->get_settings_template(
			'new-account-form',
			[
				'core_name' => esc_html__( 'AWeber', 'wpforms-aweber' ),
			]
		);
		// phpcs:enable WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}
