<?php

namespace WPFormsPaypalCommerce\Admin;

use WPFormsPaypalCommerce\Plugin;

/**
 * Extending Payments class for PayPal Commerce.
 *
 * @since 1.3.0
 */
class Payments {

	/**
	 * PayPal environment URL.
	 * Test (sandbox) mode.
	 *
	 * @since 1.3.0
	 */
	const TEST_ENV_URL = 'https://www.sandbox.paypal.com/';

	/**
	 * PayPal environment URL.
	 * Live (production) mode.
	 *
	 * @since 1.3.0
	 */
	const LIVE_ENV_URL = 'https://www.paypal.com/';

	/**
	 * Register hooks.
	 *
	 * @since 1.3.0
	 */
	public function hooks() {

		add_filter( 'wpforms_admin_payments_views_single_gateway_dashboard_link', [ $this, 'gateway_dashboard_link' ], 10, 2 );
		add_filter( 'wpforms_admin_payments_views_single_gateway_transaction_link', [ $this, 'gateway_transaction_link' ], 10, 2 );
		add_filter( 'wpforms_admin_payments_views_single_gateway_subscription_link', [ $this, 'gateway_subscription_link' ], 10, 2 );
		add_filter( 'wpforms_admin_payments_views_single_gateway_customer_link', [ $this, 'gateway_customer_link' ], 10, 2 );
		add_filter( 'wpforms_admin_payments_views_single_gateway_action_link', [ $this, 'gateway_action_link' ], 10, 3 );
	}

	/**
	 * Return gateway dashboard link.
	 *
	 * @since 1.3.0
	 *
	 * @param string $link    Dashboard link.
	 * @param object $payment Payment object.
	 *
	 * @return string
	 */
	public function gateway_dashboard_link( $link, $payment ) {

		// Return the original link if the payment is not for this gateway.
		if ( ! $this->is_this_gateway( $payment ) ) {
			return $link;
		}

		return $this->get_env_url( $payment ) . 'myaccount/summary/';
	}

	/**
	 * Return gateway transaction link.
	 *
	 * @since 1.3.0
	 *
	 * @param string $link    Dashboard link.
	 * @param object $payment Payment object.
	 *
	 * @return string
	 */
	public function gateway_transaction_link( $link, $payment ) {

		// Return the original link if the payment is not for this gateway.
		if ( ! $this->is_this_gateway( $payment ) ) {
			return $link;
		}

		return $this->get_env_url( $payment ) . "activity/payment/{$payment->transaction_id}";
	}

	/**
	 * Return gateway subscription link.
	 *
	 * @since 1.3.0
	 *
	 * @param string $link    Dashboard link.
	 * @param object $payment Payment object.
	 *
	 * @return string
	 */
	public function gateway_subscription_link( $link, $payment ) {

		// Return the original link if the payment is not for this gateway.
		if ( ! $this->is_this_gateway( $payment ) ) {
			return $link;
		}

		return $this->get_env_url( $payment ) . "billing/subscriptions/{$payment->subscription_id}";
	}

	/**
	 * Return the gateway customer link.
	 *
	 * @since 1.3.0
	 *
	 * @param string $link    Dashboard link.
	 * @param object $payment Payment object.
	 *
	 * @return string
	 */
	public function gateway_customer_link( $link, $payment ) {

		// Return the original link if the payment is not for this gateway.
		if ( ! $this->is_this_gateway( $payment ) ) {
			return $link;
		}

		return $this->gateway_dashboard_link( '', $payment );
	}

	/**
	 * Return the gateway action link.
	 *
	 * @since 1.3.0
	 *
	 * @param string $link    Dashboard link.
	 * @param string $action  Action.
	 * @param object $payment Payment object.
	 *
	 * @return string
	 */
	public function gateway_action_link( $link, $action, $payment ) {

		// Return the original link if the payment is not for this gateway.
		if ( ! $this->is_this_gateway( $payment ) ) {
			return $link;
		}

		// Return the dashboard link for all refund actions.
		if ( $action === 'refund' ) {
			return $this->gateway_transaction_link( '', $payment );
		}

		// Return the subscription link for all cancel actions.
		if ( $action === 'cancel' ) {
			return $this->gateway_subscription_link( '', $payment );
		}

		return $link;
	}

	/**
	 * Return an environment URL according to a payment mode.
	 *
	 * @since 1.3.0
	 *
	 * @param object $payment Payment object.
	 *
	 * @return string
	 */
	private function get_env_url( $payment ) {

		return isset( $payment->mode ) && $payment->mode === 'test' ? self::TEST_ENV_URL : self::LIVE_ENV_URL;
	}

	/**
	 * Check if the payment is for this gateway.
	 *
	 * @since 1.3.0
	 *
	 * @param object $payment Payment object.
	 *
	 * @return bool
	 */
	private function is_this_gateway( $payment ) {

		return isset( $payment->gateway ) && $payment->gateway === Plugin::SLUG;
	}
}
