<?php

namespace WPFormsPaypalCommerce\Process;

use WPForms\Tasks\Meta;
use WPFormsPaypalCommerce\Api\Api;
use WPFormsPaypalCommerce\Connection;
use WPFormsPaypalCommerce\Helpers;
use WPFormsPaypalCommerce\Plugin;

/**
 * PayPal Commerce payment processing.
 *
 * @since 1.0.0
 */
class Process extends Base {

	/**
	 * Task name to update subscription payment.
	 *
	 * @since 1.3.0
	 *
	 * @var string
	 */
	const SUBSCRIPTION_TASK = 'wpforms_paypal_commerce_subscription_payment_data_update';

	/**
	 * PayPal Commerce field.
	 *
	 * @since 1.0.0
	 *
	 * @var array
	 */
	private $field = [];

	/**
	 * Form submission data ($_POST).
	 *
	 * @since 1.0.0
	 *
	 * @var array
	 */
	private $entry = [];

	/**
	 * Main class that communicates with the PayPal Commerce API.
	 *
	 * @since 1.0.0
	 *
	 * @var Api
	 */
	protected $api;

	/**
	 * Whether the payment has been processed.
	 *
	 * @since 1.5.0
	 *
	 * @var bool
	 */
	private $is_payment_processed = false;

	/**
	 * Register hooks.
	 *
	 * @since 1.0.0
	 */
	public function hooks() {

		if ( wp_doing_ajax() ) {
			( new ProcessSingleAjax() )->hooks();
			( new ProcessSubscriptionAjax() )->hooks();
		}

		add_action( 'wpforms_process', [ $this, 'process_entry' ], 10, 3 );
		add_action( 'wpforms_process_complete', [ $this, 'update_entry_meta' ], 10, 4 );
		add_filter( 'wpforms_entry_email_process', [ $this, 'process_email' ], 70, 5 );
		add_filter( 'wpforms_forms_submission_prepare_payment_data', [ $this, 'prepare_payment_data' ], 10, 3 );
		add_filter( 'wpforms_forms_submission_prepare_payment_meta', [ $this, 'prepare_payment_meta' ], 10, 3 );
		add_action( 'wpforms_process_payment_saved', [ $this, 'process_payment_saved' ], 10, 3 );
		add_action( self::SUBSCRIPTION_TASK, [ $this, 'update_subscription_data_scheduled_task' ] );
		add_filter( 'wpforms_process_bypass_captcha', [ $this, 'bypass_captcha' ] );
	}

	/**
	 * Check if a payment exists with an entry, if so validate and process.
	 *
	 * @since 1.0.0
	 *
	 * @param array $fields    Final/sanitized submitted field data.
	 * @param array $entry     Copy of original $_POST.
	 * @param array $form_data Form data and settings.
	 */
	public function process_entry( $fields, $entry, $form_data ) {

		if ( ! Helpers::is_paypal_commerce_enabled( $form_data ) ) {
			return;
		}

		$this->form_data  = $form_data;
		$this->fields     = $fields;
		$this->entry      = $entry;
		$this->form_id    = (int) $form_data['id'];
		$this->amount     = $this->get_amount();
		$this->field      = Helpers::get_paypal_field( $this->fields );
		$this->connection = Connection::get();
		$this->api        = wpforms_paypal_commerce()->get_api( $this->connection );

		if ( is_null( $this->api ) ) {
			return;
		}

		// Before proceeding, check if any basic errors were detected.
		if ( ! $this->is_form_ok() || ! $this->is_form_processed() ) {
			$this->display_errors();

			return;
		}

		if (
			empty( $entry['fields'][ $this->field['id'] ]['orderID'] )
			&& empty( $entry['fields'][ $this->field['id'] ]['subscriptionID'] )
		) {
			$this->display_errors();

			return;
		}
		// Set payment processing flag.
		$this->is_payment_processed = true;

		if ( ! empty( $entry['fields'][ $this->field['id'] ]['orderID'] ) ) {
			$this->capture_single();

			return;
		}

		$this->activate_subscription();
	}

	/**
	 * Bypass captcha if payment has been processed.
	 *
	 * @since 1.5.0
	 *
	 * @param bool $bypass_captcha Whether to bypass captcha.
	 *
	 * @return bool
	 */
	public function bypass_captcha( $bypass_captcha ) {

		if ( $bypass_captcha ) {
			return $bypass_captcha;
		}

		return $this->is_payment_processed;
	}

	/**
	 * Capture single order.
	 *
	 * @since 1.0.0
	 */
	private function capture_single() {

		$order_response = $this->api->capture( $this->entry['fields'][ $this->field['id'] ]['orderID'] );

		if ( $order_response->has_errors() ) {
			$error_title    = esc_html__( 'This payment cannot be processed because there was an error with the capture order API call.', 'wpforms-paypal-commerce' );
			$this->errors[] = $error_title;

			$this->log_errors( $error_title, $order_response->get_response_message() );

			return;
		}

		$order_data = $order_response->get_body();

		if ( isset( $order_data['payment_source']['card'] ) ) {
			wpforms()->get( 'process' )->fields[ $this->field['id'] ]['value'] = implode( "\n", array_filter( $order_data['payment_source']['card'] ) );
		} else {
			wpforms()->get( 'process' )->fields[ $this->field['id'] ]['value'] = '-';
		}
	}

	/**
	 * Activate subscription.
	 *
	 * @since 1.0.0
	 */
	private function activate_subscription() {

		$subscription_id = $this->entry['fields'][ $this->field['id'] ]['subscriptionID'];

		$subscription_response = $this->api->activate_subscription( $subscription_id );

		if ( $subscription_response->has_errors() ) {
			$error_title    = esc_html__( 'This subscription cannot be activated because there was an error with the activation API call.', 'wpforms-paypal-commerce' );
			$this->errors[] = $error_title;

			$this->log_errors( $error_title, $subscription_response->get_response_message() );
		} else {
			wpforms()->get( 'process' )->fields[ $this->field['id'] ]['value'] = '-';
		}
	}

	/**
	 * Update entry details and add meta for a successful payment.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $fields    Final/sanitized submitted field data.
	 * @param array  $entry     Copy of original $_POST.
	 * @param array  $form_data Form data and settings.
	 * @param string $entry_id  Entry ID.
	 */
	public function update_entry_meta( $fields, $entry, $form_data, $entry_id ) {

		if ( empty( $entry_id ) || $this->errors || ! $this->api || empty( $this->field ) || ( empty( $entry['fields'][ $this->field['id'] ]['orderID'] ) && empty( $entry['fields'][ $this->field['id'] ]['subscriptionID'] ) ) ) {
			return;
		}

		$order_data = $this->get_order_data();

		if ( empty( $order_data ) ) {
			$order_data = $this->get_subscription_data();
		}

		// If we don't have order data, bail.
		if ( empty( $order_data ) ) {
			return;
		}

		wpforms()->get( 'entry' )->update(
			$entry_id,
			[
				'type' => 'payment',
			],
			'',
			'',
			[ 'cap' => false ]
		);

		/**
		 * Fire when entry details and add meta was successfully updated.
		 *
		 * @since 1.0.0
		 *
		 * @param array   $fields     Final/sanitized submitted field data.
		 * @param array   $form_data  Form data and settings.
		 * @param string  $entry_id   Entry ID.
		 * @param array   $order_data Response order data.
		 * @param Process $process    Process class instance.
		 */
		do_action( 'wpforms_paypal_commerce_process_update_entry_meta', $fields, $form_data, $entry_id, $order_data, $this );
	}

	/**
	 * Logic that helps decide if we should send completed payments notifications.
	 *
	 * @since 1.0.0
	 *
	 * @param bool   $process         Whether to process or not.
	 * @param array  $fields          Form fields.
	 * @param array  $form_data       Form data.
	 * @param int    $notification_id Notification ID.
	 * @param string $context         In which context this email is sent.
	 *
	 * @return bool
	 */
	public function process_email( $process, $fields, $form_data, $notification_id, $context ) {

		if ( ! $process ) {
			return false;
		}

		if ( ! Helpers::is_paypal_commerce_enabled( $form_data ) ) {
			return $process;
		}

		if ( empty( $form_data['settings']['notifications'][ $notification_id ][ Plugin::SLUG ] ) ) {
			return $process;
		}

		if ( empty( $this->entry['fields'][ $this->field['id'] ]['orderID'] ) && empty( $this->entry['fields'][ $this->field['id'] ]['subscriptionID'] ) ) {
			return false;
		}

		return ! $this->errors && $this->api;
	}

	/**
	 * Get order data.
	 *
	 * @since 1.3.0
	 *
	 * @return array
	 */
	private function get_order_data() {

		// If the payment processing is not allowed, bail.
		if ( ! $this->is_payment_saving_allowed() ) {
			return [];
		}

		static $order_data;

		if ( ! is_null( $order_data ) ) {
			return $order_data;
		}

		if ( empty( $this->entry['fields'][ $this->field['id'] ]['orderID'] ) ) {
			return [];
		}

		$order_data = $this->api->get_order( $this->entry['fields'][ $this->field['id'] ]['orderID'] );

		return $order_data;
	}

	/**
	 * Get order data.
	 *
	 * @since 1.3.0
	 *
	 * @return array
	 */
	private function get_subscription_data() {

		// If the payment processing is not allowed, bail.
		if ( ! $this->is_payment_saving_allowed() ) {
			return [];
		}

		static $subscription_data;

		if ( ! is_null( $subscription_data ) ) {
			return $subscription_data;
		}

		if ( empty( $this->entry['fields'][ $this->field['id'] ]['subscriptionID'] ) ) {
			return [];
		}

		$subscription_data = $this->api->get_subscription( $this->entry['fields'][ $this->field['id'] ]['subscriptionID'] );

		return $subscription_data;
	}

	/**
	 * Add details to payment data.
	 *
	 * @since 1.3.0
	 *
	 * @param array $payment_data Payment data args.
	 * @param array $fields       Form fields.
	 * @param array $form_data    Form data.
	 *
	 * @return array
	 */
	public function prepare_payment_data( $payment_data, $fields, $form_data ) {

		// Determine whether this is a one-time payment.
		$order_data = $this->get_order_data();

		if ( ! empty( $order_data ) ) {

			$payment_data['transaction_id'] = sanitize_text_field( $order_data['purchase_units'][0]['payments']['captures'][0]['id'] );
			$payment_data['title']          = $this->get_payment_title( $order_data, $form_data );

			return $this->add_generic_payment_data( $payment_data );
		}

		// Determine whether it is a subscription.
		$subscription_data = $this->get_subscription_data();

		if ( ! empty( $subscription_data ) ) {

			$payment_data['subscription_status'] = 'not-synced';
			$payment_data['subscription_id']     = sanitize_text_field( $subscription_data['id'] );
			$payment_data['customer_id']         = sanitize_text_field( $subscription_data['subscriber']['payer_id'] );
			$payment_data['title']               = $this->get_payment_title( $subscription_data, $form_data );

			$this->maybe_log_matched_subscriptions( $subscription_data['plan_id'] );

			return $this->add_generic_payment_data( $payment_data );
		}

		return $payment_data;
	}

	/**
	 * Get Payment title.
	 *
	 * @since 1.3.0
	 *
	 * @param array $order_data Order data.
	 * @param array $form_data  Form data.
	 *
	 * @return string Payment title.
	 */
	private function get_payment_title( $order_data, $form_data ) {

		if ( ! empty( $this->entry['fields'][ $this->field['id'] ]['cardname'] ) ) {
			return sanitize_text_field( $this->entry['fields'][ $this->field['id'] ]['cardname'] );
		}

		$customer_name = $this->get_customer_name( $order_data, $form_data );

		if ( $customer_name ) {
			return sanitize_text_field( $customer_name );
		}

		if ( ! empty( $order_data['purchase_units'][0]['payee']['email_address'] ) ) {
			return sanitize_email( $order_data['purchase_units'][0]['payee']['email_address'] );
		}

		if ( ! empty( $order_data['subscriber']['email_address'] ) ) {
			return sanitize_email( $order_data['subscriber']['email_address'] );
		}

		if ( ! empty( $this->fields[ $form_data['payments'][ Plugin::SLUG ]['billing_email'] ]['value'] ) ) {
			return $this->fields[ $form_data['payments'][ Plugin::SLUG ]['billing_email'] ]['value'];
		}

		return '';
	}

	/**
	 * Add payment meta for a successful one-time or subscription payment.
	 *
	 * @since 1.3.0
	 *
	 * @param array $payment_meta Payment meta.
	 * @param array $fields       Sanitized submitted field data.
	 * @param array $form_data    Form data and settings.
	 *
	 * @return array
	 */
	public function prepare_payment_meta( $payment_meta, $fields, $form_data ) {

		// Retrieve order data for one-time payments.
		$order_data = $this->get_order_data();

		if ( ! empty( $order_data ) ) {
			$payment_meta                = $this->add_credit_card_meta( $payment_meta, $order_data );
			$payment_meta['method_type'] = sanitize_text_field( $this->get_payment_method_type() );
			$payment_meta['log']         = $this->format_payment_log(
				sprintf(
					'PayPal Commerce Order created. (Order ID: %s)',
					$order_data['id']
				)
			);

			return $payment_meta;
		}

		// Retrieve subscription data.
		$subscription_data = $this->get_subscription_data();

		if ( ! empty( $subscription_data ) ) {
			$payment_meta['method_type']         = 'checkout';
			$payment_meta['subscription_period'] = $this->get_subscription_period( $form_data, $subscription_data['plan_id'] );
			$payment_meta['log']                 = $this->format_payment_log(
				sprintf(
					'PayPal Commerce Subscription created. (Subscription ID: %s)',
					$subscription_data['id']
				)
			);

			return $payment_meta;
		}

		// If no order or subscription data was found, return the payment meta.
		return $payment_meta;
	}

	/**
	 * Add payment info for successful payment.
	 *
	 * @since 1.3.0
	 *
	 * @param string $payment_id Payment ID.
	 * @param array  $fields     Final/sanitized submitted field data.
	 * @param array  $form_data  Form data and settings.
	 */
	public function process_payment_saved( $payment_id, $fields, $form_data ) {

		// Determine whether this is a subscription payment.
		$subscription_data = $this->get_subscription_data();

		if ( ! empty( $subscription_data ) ) {
			$this->schedule_subscription_update( $payment_id, $subscription_data['id'] );

			return;
		}

		// Determine whether this is a one-time payment.
		$order_data = $this->get_order_data();

		if ( empty( $order_data ) ) {
			return;
		}

		$this->add_completed_log( $payment_id, $order_data['purchase_units'][0]['payments']['captures'][0]['id'] );
	}

	/**
	 * Schedule update subscription due to some delay in PayPal API.
	 *
	 * @since 1.3.0
	 *
	 * @param int $payment_id      Payment ID.
	 * @param int $subscription_id Subscription ID.
	 */
	private function schedule_subscription_update( $payment_id, $subscription_id ) {

		$tasks = wpforms()->get( 'tasks' );

		$tasks->create( self::SUBSCRIPTION_TASK )
			->params( $payment_id, $subscription_id )
			->once( time() + 60 )
			->register();
	}

	/**
	 * Update subscription transaction ID in task due to some delay in PayPal API.
	 *
	 * @since 1.3.0
	 *
	 * @param int $meta_id Action meta id.
	 */
	public function update_subscription_data_scheduled_task( $meta_id ) {

		$params = ( new Meta() )->get( $meta_id );

		if ( ! $params ) {
			return;
		}

		list( $payment_id, $subscription_id ) = $params->data;

		$transactions   = wpforms_paypal_commerce()->get_api( Connection::get() )->get_subscription_transactions( $subscription_id );
		$transaction_id = $transactions ? end( $transactions )['id'] : '';

		$this->add_completed_log( $payment_id, $transaction_id );

		wpforms()->get( 'payment' )->update( $payment_id, [ 'transaction_id' => $transaction_id ], '', '', [ 'cap' => false ] );
	}

	/**
	 * Add completed payment log.
	 *
	 * @since 1.3.0
	 *
	 * @param string $payment_id     Payment id.
	 * @param string $transaction_id Transaction id.
	 */
	private function add_completed_log( $payment_id, $transaction_id ) {

		// phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_key, WordPress.DB.SlowDBQuery.slow_db_query_meta_value
		wpforms()->get( 'payment_meta' )->add(
			[
				'payment_id' => $payment_id,
				'meta_key'   => 'log',
				'meta_value' => $this->format_payment_log(
					sprintf(
						'PayPal Commerce payment completed. (Transaction ID: %s)',
						$transaction_id
					)
				),
			]
		);
		// phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_key, WordPress.DB.SlowDBQuery.slow_db_query_meta_value
	}

	/**
	 * Return payment log value.
	 *
	 * @since 1.3.0
	 *
	 * @param string $value Log value.
	 *
	 * @return string
	 */
	private function format_payment_log( $value ) {

		return wp_json_encode(
			[
				'value' => sanitize_text_field( $value ),
				'date'  => gmdate( 'Y-m-d H:i:s' ),
			]
		);
	}

	/**
	 * Determine the payment method name.
	 * If PayPal, return 'checkout', otherwise 'card'.
	 *
	 * @since 1.3.0
	 *
	 * @return string
	 */
	private function get_payment_method_type() {

		return $this->entry['fields'][ $this->field['id'] ]['source'] === 'paypal' ? 'checkout' : 'card';
	}

	/**
	 * Get Customer name.
	 *
	 * @since 1.3.0
	 *
	 * @param array $order_data Order data.
	 * @param array $form_data  Form data and settings.
	 *
	 * @return string
	 */
	private function get_customer_name( $order_data, $form_data ) {

		if ( ! empty( $order_data['payer']['name'] ) ) {
			return implode( ' ', array_values( $order_data['payer']['name'] ) );
		}

		if ( ! empty( $order_data['subscriber']['name'] ) ) {
			return implode( ' ', array_values( $order_data['subscriber']['name'] ) );
		}

		$customer_name = [];
		$form_settings = $form_data['payments'][ Plugin::SLUG ];

		// Billing first name.
		if ( ! empty( $this->fields[ $form_settings['name'] ]['first'] ) ) {
			$customer_name['first_name'] = $this->fields[ $form_settings['name'] ]['first'];
		}

		// Billing last name.
		if ( ! empty( $this->fields[ $form_settings['name'] ]['last'] ) ) {
			$customer_name['last_name'] = $this->fields[ $form_settings['name'] ]['last'];
		}

		if (
			empty( $customer_name['first_name'] ) &&
			empty( $customer_name['last_name'] ) &&
			! empty( $this->fields[ $form_settings['name'] ]['value'] )
		) {
			$customer_name['first_name'] = $this->fields[ $form_settings['name'] ]['value'];
		}

		return implode( ' ', array_values( $customer_name ) );
	}

	/**
	 * Add generic payment data.
	 *
	 * @since 1.3.0
	 *
	 * @param array $payment_data Payment data.
	 *
	 * @return array
	 */
	private function add_generic_payment_data( $payment_data ) {

		$payment_data['status']  = 'processed';
		$payment_data['gateway'] = Plugin::SLUG;
		$payment_data['mode']    = Helpers::is_sandbox_mode() ? 'test' : 'live';

		return $payment_data;
	}

	/**
	 * Add credit card meta.
	 *
	 * @since 1.3.0
	 *
	 * @param array $payment_meta Payment meta.
	 * @param array $order_data   Order data.
	 *
	 * @return array
	 */
	private function add_credit_card_meta( $payment_meta, $order_data ) {

		// Bail early if payment source is not available.
		if ( empty( $order_data['payment_source'] ) ) {
			return $payment_meta;
		}

		$payment_source = $order_data['payment_source'];

		// Add credit card holder name, e.g. John Doe.
		if ( ! empty( $this->entry['fields'][ $this->field['id'] ]['cardname'] ) ) {
			$payment_meta['credit_card_name'] = sanitize_text_field( $this->entry['fields'][ $this->field['id'] ]['cardname'] );
		}

		// Add credit card brand name, e.g. Visa, MasterCard, etc.
		if ( ! empty( $payment_source['card']['brand'] ) ) {
			$payment_meta['credit_card_method'] = sanitize_text_field( strtolower( $payment_source['card']['brand'] ) );
		}

		// Add credit card last 4 digits, e.g. 1234, 5678, etc.
		if ( ! empty( $payment_source['card']['last_digits'] ) ) {
			$payment_meta['credit_card_last4'] = sanitize_text_field( $payment_source['card']['last_digits'] );
		}

		// Add credit card expiry date, e.g. 2029-11, 2024-10, etc.
		if ( ! empty( $payment_source['card']['expiry'] ) ) {
			$payment_meta['credit_card_expires'] = sanitize_text_field( $payment_source['card']['expiry'] );
		}

		return $payment_meta;
	}

	/**
	 * Get subscription period by plan id.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $form_data  Form data.
	 * @param string $pp_plan_id Subscription plan id.
	 *
	 * @return string
	 */
	private function get_subscription_period( $form_data, $pp_plan_id ) {

		foreach ( $form_data['payments'][ Plugin::SLUG ]['recurring'] as $recurring ) {

			if ( $recurring['pp_plan_id'] !== $pp_plan_id ) {
				continue;
			}

			return str_replace( '-', '', $recurring['recurring_times'] );
		}

		return '';
	}

	/**
	 * Log if more than one plan matched on form submission.
	 *
	 * @since 1.0.0
	 *
	 * @param string $matched_plan_id Already matched and executed plan.
	 */
	private function maybe_log_matched_subscriptions( $matched_plan_id ) {

		foreach ( $this->form_data['payments'][ Plugin::SLUG ]['recurring'] as $recurring ) {

			if ( ! $this->is_conditional_logic_ok( $recurring ) || $recurring['pp_plan_id'] === $matched_plan_id ) {
				continue;
			}

			$this->log_errors(
				'PayPal Commerce subscription processing error.',
				sprintf(
					/* translators: %1$s - Plan ID, %2$s - Plan ID. */
					esc_html( 'Plan %1$s processing error. Plan %2$s already matched.' ),
					$recurring['pp_plan_id'],
					$matched_plan_id
				)
			);
		}
	}

	/**
	 * Check if form has errors before payment processing.
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	private function is_form_processed() {

		// Bail in case there are form processing errors.
		if ( ! empty( wpforms()->get( 'process' )->errors[ $this->form_id ] ) ) {
			return false;
		}

		return $this->is_card_field_visibility_ok();
	}

	/**
	 * Check if there is at least one visible (not hidden by conditional logic) card field in the form.
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	private function is_card_field_visibility_ok() {

		if ( empty( $this->field ) ) {
			return false;
		}

		// If the form contains no fields with conditional logic the card field is visible by default.
		if ( empty( $this->form_data['conditional_fields'] ) ) {
			return true;
		}

		// If the field is NOT in array of conditional fields, it's visible.
		if ( ! in_array( $this->field['id'], $this->form_data['conditional_fields'], true ) ) {
			return true;
		}

		// If the field IS in array of conditional fields and marked as visible, it's visible.
		if ( ! empty( $this->field['visible'] ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Display form errors.
	 *
	 * @since 1.0.0
	 */
	private function display_errors() {

		if ( ! $this->errors || ! is_array( $this->errors ) ) {
			return;
		}

		// Check if the form contains a required credit card. If it does
		// and there was an error, return the error to the user and prevent
		// the form from being submitted. This should not occur under normal
		// circumstances.
		if ( empty( $this->field ) || empty( $this->form_data['fields'][ $this->field['id'] ] ) ) {
			return;
		}

		if ( ! empty( $this->form_data['fields'][ $this->field['id'] ]['required'] ) ) {
			wpforms()->get( 'process' )->errors[ $this->form_id ]['footer'] = implode( '<br>', $this->errors );
		}
	}

	/**
	 * Determine if payment saving is allowed, by checking if the form has a payment field, and the API is available.
	 *
	 * @since 1.3.0
	 *
	 * @return bool
	 */
	private function is_payment_saving_allowed() {

		return ! empty( $this->field ) && $this->api;
	}
}
