<?php

namespace WPFormsPaypalStandard\Admin;

/**
 * Extending Payments class for PayPal Standard.
 *
 * @since 1.7.0
 */
class Payments {

	/**
	 * PayPal Standard environment URL.
	 * Test (sandbox) mode.
	 *
	 * @since 1.7.0
	 */
	const TEST_ENV_URL = 'https://www.sandbox.paypal.com/';

	/**
	 * PayPal Standard environment URL.
	 * Live (production) mode.
	 *
	 * @since 1.7.0
	 */
	const LIVE_ENV_URL = 'https://www.paypal.com/';

	/**
	 * PayPal Standard gateway ID.
	 *
	 * @since 1.7.0
	 */
	const GATEWAY_ID = 'paypal_standard';

	/**
	 * Register hooks.
	 *
	 * @since 1.7.0
	 */
	public function hooks() {

		add_filter( 'wpforms_admin_payments_views_single_gateway_dashboard_link', [ $this, 'gateway_dashboard_link' ], 10, 2 );
		add_filter( 'wpforms_admin_payments_views_single_gateway_transaction_link', [ $this, 'gateway_transaction_link' ], 10, 2 );
		add_filter( 'wpforms_admin_payments_views_single_gateway_customer_link', [ $this, 'gateway_customer_link' ], 10, 2 );
		add_filter( 'wpforms_admin_payments_views_single_gateway_action_link', [ $this, 'gateway_action_link' ], 10, 3 );
	}

	/**
	 * Return gateway dashboard link.
	 *
	 * @since 1.7.0
	 *
	 * @param string $link    Dashboard link.
	 * @param object $payment Payment object.
	 *
	 * @return string
	 */
	public function gateway_dashboard_link( $link, $payment ) {

		// Return the original link if the payment is not for this gateway.
		if ( ! $this->is_this_gateway( $payment ) ) {
			return $link;
		}

		return $this->get_env_url( $payment ) . 'myaccount/summary/';
	}

	/**
	 * Return gateway transaction link.
	 *
	 * @since 1.7.0
	 *
	 * @param string $link    Dashboard link.
	 * @param object $payment Payment object.
	 *
	 * @return string
	 */
	public function gateway_transaction_link( $link, $payment ) {

		// Return the original link if the payment is not for this gateway.
		if ( ! $this->is_this_gateway( $payment ) ) {
			return $link;
		}

		return $this->get_env_url( $payment ) . "activity/payment/{$payment->transaction_id}";
	}

	/**
	 * Return the gateway customer link.
	 *
	 * @since 1.7.0
	 *
	 * @param string $link    Dashboard link.
	 * @param object $payment Payment object.
	 *
	 * @return string
	 */
	public function gateway_customer_link( $link, $payment ) {

		// Return the original link if the payment is not for this gateway.
		if ( ! $this->is_this_gateway( $payment ) ) {
			return $link;
		}

		return $this->gateway_dashboard_link( '', $payment );
	}

	/**
	 * Return the gateway action link.
	 *
	 * @since 1.7.0
	 *
	 * @param string $link    Dashboard link.
	 * @param string $action  Action.
	 * @param object $payment Payment object.
	 *
	 * @return string
	 */
	public function gateway_action_link( $link, $action, $payment ) {

		// Return the original link if the payment is not for this gateway.
		if ( ! $this->is_this_gateway( $payment ) ) {
			return $link;
		}

		// Return the dashboard link for all refund actions.
		if ( $action === 'refund' ) {
			return $this->gateway_transaction_link( '', $payment );
		}

		return $link;
	}

	/**
	 * Return an environment URL according to a payment mode.
	 *
	 * @since 1.7.0
	 *
	 * @param object $payment Payment object.
	 *
	 * @return string
	 */
	private function get_env_url( $payment ) {

		return isset( $payment->mode ) && $payment->mode === 'test' ? self::TEST_ENV_URL : self::LIVE_ENV_URL;
	}

	/**
	 * Check if the payment is for this gateway.
	 *
	 * @since 1.7.0
	 *
	 * @param object $payment Payment object.
	 *
	 * @return bool
	 */
	private function is_this_gateway( $payment ) {

		return $payment->gateway === self::GATEWAY_ID;
	}
}
